import { FamilyUser } from '@client/shared/api/graphql';
import { getAvatarUrl } from '@client/shared/paths';
import { i18nServices } from '@client/shared/translations/services';
import { formatPlus, formatPrice } from '@shared/helpers';

import { PaymentItemProps } from '../components/PaymentItem';
import { Order } from '../types';
import { getDescription } from './description';
import { isPlusTopup } from './order';
import { getServiceName } from './serviceName';

type Options = {
  plusServiceIconUrl?: string;
  familyData?: FamilyUser[] | null;
  avatarHost: string;
};

export const dataToProps = (data: Order, options: Options): PaymentItemProps => {
  const { service, total, plus } = data;
  const { plusServiceIconUrl, familyData, avatarHost } = options;

  const isPlus = isPlusTopup(data);
  const isRefunded = data.status === 'refunded';
  let label = data.retailerName || (isPlus ? i18nServices('plus_points') : getServiceName(data));

  if (data.source === 'yandexpay') {
    label = i18nServices('yandexpay_purchase');
  }

  let iconUrl = data.retailerIcon || (isPlus ? plusServiceIconUrl : service.iconUrl);
  let subIconUrl;
  let labelPrefix;

  if (data.initiatorUid) {
    const user = familyData?.find(({ uid }) => uid === data.initiatorUid);

    labelPrefix = user?.name;

    if (user && user.avatar) {
      subIconUrl = iconUrl;
      iconUrl = getAvatarUrl(avatarHost, user.avatar);
    }
  }

  return {
    label,
    labelPrefix,
    iconUrl,
    text: isPlus ? getServiceName(data) : getDescription(data, { useDefault: true }),
    cash:
      !isPlus && total ? formatPrice(total, data.currency, data.status === 'refunded') : undefined,
    plus: plus ? formatPlus(plus, isRefunded, { leadPlus: true }) : undefined,
    status: data.status,
    edadeal: data.cashback
      ? formatPrice(data.cashback, data.currency, true, { signed: true })
      : undefined,
    subIconUrl,
  };
};
