import { isPlusPaymentMethod } from '@shared/helpers';

import { Order } from '../types';

export const isPlusTopup = ({ paymentTypes, status }: Order) => {
  return (
    (paymentTypes &&
      paymentTypes.length &&
      paymentTypes?.every((payment) => payment.method === 'yandex_account_topup')) ||
    (status === 'refunded' &&
      paymentTypes?.every((payment) => payment.method === 'yandex_account_withdraw'))
  );
};

export const isPlusTransaction = ({ plus, paymentTypes }: Order) => {
  return Boolean(
    plus &&
      paymentTypes &&
      paymentTypes.length === 1 &&
      isPlusPaymentMethod(paymentTypes[0].method),
  );
};

export type PaymentSystem =
  | 'Mir'
  | 'AmericanExpress'
  | 'Visa'
  | 'MasterCard'
  | 'UnionPay'
  | 'UnknownCard';

export type PlatformPay = 'google_pay' | 'apple_pay';

const numberToPaymentSystem: Record<string, PaymentSystem> = {
  '2': 'Mir',
  '3': 'AmericanExpress',
  '4': 'Visa',
  '5': 'MasterCard',
  '6': 'UnionPay',
};

export function getPaymentSystemByMask(mask = ''): PaymentSystem {
  return numberToPaymentSystem[mask.charAt(0)] || 'UnknownCard';
}

const knownPaymentMethods = [
  'card',
  'yamoney_wallet',
  'cash',
  'new_promocode',
  'spasibo',
  'yandex_account_withdraw',
  'yandex_account_topup',
  'afisha_certificate',
  'google_pay',
  'apple_pay',
  'virtual_kinopoisk',
  'cashback_edadeal',
];

export const getPaymentMethodKey = (method: string): string => {
  let _method: string = method;

  switch (method) {
    case 'virtual::kinopoisk_card_discounts':
    case 'virtual::kinopoisk_subs_discounts':
      _method = 'virtual_kinopoisk';

      break;
    case 'virtual::new_promocode':
      _method = 'new_promocode';

      break;
    case 'spasibo_cashback':
      _method = 'spasibo';

      break;
  }

  if (knownPaymentMethods.indexOf(_method) >= 0) {
    return _method;
  }

  return 'paying';
};
