import { FC, RefObject, useCallback, useMemo, useRef } from 'react';

import { PersonalDataModal, PersonalDataModel } from '@client/features/personal-data';
import { AvatarCropModal } from '@client/features/personal-data';
import { useProfileWidgetQuery } from '@client/shared/api/graphql';
import { logProps } from '@client/shared/libs/metrika';
import { useAvatarsPath } from '@client/shared/paths';
import { Section, UserInfo, UserInfoProps, UserInfoSkeleton } from '@client/shared/ui-kit';

function formatDate(rawDate: string) {
  const result = /(\d{4})-(\d{2})-(\d{2})/.exec(rawDate);

  return `${result?.[3]}.${result?.[2]}.${result?.[1]}`;
}

interface UserInfoComponentProps {
  innerRef: RefObject<HTMLDivElement>;
  onPress: UserInfoProps['onPress'];
}

const UserInfoComponent: FC<UserInfoComponentProps> = (props) => {
  const { loading, data, error } = useProfileWidgetQuery();
  const avatarsHost = useAvatarsPath();

  const { innerRef, onPress } = props;

  const fullName = useMemo(() => {
    const firstName = data?.viewer?.firstName || '';
    const lastName = data?.viewer?.lastName || '';

    if (firstName || lastName) {
      return `${firstName} ${lastName}`.trim();
    }
  }, [data?.viewer?.firstName, data?.viewer?.lastName]);

  if (loading) {
    return <UserInfoSkeleton />;
  }

  if (error || !data) {
    return null;
  }

  const { viewer } = data;
  const { displayName, birthdate, avatarId, isChild, havePlus, city } = viewer;

  return (
    <UserInfo
      {...logProps('userinfo.open-menu')}
      ref={innerRef}
      displayName={displayName}
      fullName={fullName}
      birthdate={birthdate ? formatDate(birthdate) : undefined}
      onPress={onPress}
      avatar={avatarId !== '0/0-0' ? `${avatarsHost}/get-yapic/${avatarId}/islands-200` : undefined}
      isChild={isChild}
      hasPlus={havePlus}
      city={city || undefined}
    />
  );
};

export const Profile = () => {
  const onUserInfoPress = useCallback(() => {
    PersonalDataModel.show();
  }, []);

  const userInfoRef = useRef<HTMLDivElement>(null);

  return (
    <Section>
      <UserInfoComponent innerRef={userInfoRef} onPress={onUserInfoPress} />
      <PersonalDataModal />
      <AvatarCropModal />
    </Section>
  );
};
