import { VFC } from 'react';

import { useProfileFillQuery } from '@client/shared/api/graphql';
import { logProps } from '@client/shared/libs/metrika';
import { useIdPaths, usePassportPaths } from '@client/shared/paths';
import {
  Carousel,
  CarouselItem,
  ProgressCircle,
  Section,
  Shortcut,
  ShortcutLink,
} from '@client/shared/ui-kit';

import { i18n } from './i18n';

export const ProfileFill: VFC = () => {
  const { data, loading, error } = useProfileFillQuery();
  const passportPaths = usePassportPaths();
  const idPaths = useIdPaths();

  if (loading || error || !data) {
    return null;
  }

  const { account, viewer, addresses } = data;

  const hasPhone = account.phones.length > 0;
  const hasAvatarAndFirstname = Boolean(viewer.firstName) && viewer.avatarId !== '0/0-0';
  const hasBirthdate = Boolean(viewer.birthdate);
  const hasAddresses = addresses.length > 0;
  const totalFlags = [hasPhone, hasAvatarAndFirstname, hasBirthdate, hasAddresses];
  const filledFlags = totalFlags.filter((v) => v);

  if (totalFlags.length === filledFlags.length) {
    return null;
  }

  return (
    <Section>
      <Section.Title anchor="profile-fill">
        {i18n('Заполните профиль', { context: 'Заголовок блока заполнения профиля в ЛК' })}
      </Section.Title>
      <Section.Description>
        {i18n('Несколько шагов помогут получить максимум от сервисов Яндекса', {
          context: 'Описание блока заполнения профиля в ЛК',
        })}
      </Section.Description>
      <Section.Aside>
        <ProgressCircle
          variant="default"
          formatStyle="decimal"
          strokeWidth={3}
          size={40}
          max={totalFlags.length}
          value={filledFlags.length}
        />
      </Section.Aside>
      <Carousel>
        <CarouselItem>
          <ShortcutLink
            href={passportPaths.social}
            size="s"
            buttonText={i18n('Заполнить', {
              context: 'Текст кнопки в шорткате заполнения профиля из соцсетей в ЛК',
            })}
            text={i18n('Взять данные из соцсети', {
              context: 'Описание шортката заполнения профиля из соцсетей в ЛК',
            })}
            label={i18n('Заполнить в один клик', {
              context: 'Заголовок шортката заполнения профиля из соцсетей в ЛК',
            })}
            labelBold
            withImage
            variant="wand"
            tall
            color="info"
            {...logProps('shortcut-link.fill-personal-data-from-social')}
          />
        </CarouselItem>
        {!hasAvatarAndFirstname && (
          <CarouselItem>
            <ShortcutLink
              href={idPaths.personalData}
              size="s"
              buttonText={i18n('Добавить', {
                context: 'Текст кнопки в шорткате заполнения профиля в ЛК',
              })}
              text={i18n('Как к вам обращаться?', {
                context: 'Описание шортката ФИО и аватара в блоке заполнения профиля в ЛК',
              })}
              label={i18n('Имя и фото', {
                context: 'Заголовок шортката ФИО и аватара в блоке заполнения профиля в ЛК',
              })}
              labelBold
              withImage
              variant="camera"
              tall
              color="info"
              {...logProps('shortcut-link.add-name-and-data')}
            />
          </CarouselItem>
        )}
        {!hasBirthdate && (
          <CarouselItem>
            <ShortcutLink
              href={idPaths.personalData}
              size="s"
              buttonText={i18n('Добавить', {
                context: 'Текст кнопки в шорткате заполнения профиля в ЛК',
              })}
              text={i18n('Чтобы вовремя поздравить', {
                context: 'Описание шортката дня рождения в блоке заполнения профиля в ЛК',
              })}
              label={i18n('День рождения', {
                context: 'Заголовок шортката дня рождения в блоке заполнения профиля в ЛК',
              })}
              labelBold
              withImage
              variant="salute"
              tall
              color="info"
              {...logProps('shortcut-link.add-birthdate')}
            />
          </CarouselItem>
        )}
        {!hasPhone && (
          <CarouselItem>
            <ShortcutLink
              href={passportPaths.phones}
              size="s"
              buttonText={i18n('Добавить', {
                context: 'Текст кнопки в шорткате заполнения профиля в ЛК',
              })}
              text={i18n('Поможет усилить защиту данных', {
                context: 'Описание шортката телефона в блоке заполнения профиля в ЛК',
              })}
              label={i18n('Телефон', {
                context: 'Заголовок шортката телефона в блоке заполнения профиля в ЛК',
              })}
              labelBold
              withImage
              variant="phone"
              tall
              color="info"
              {...logProps('shortcut-link.add-phone')}
            />
          </CarouselItem>
        )}
        {!hasAddresses && (
          <CarouselItem>
            <ShortcutLink
              href={idPaths.addresses}
              size="s"
              buttonText={i18n('Добавить', {
                context: 'Текст кнопки в шорткате заполнения профиля в ЛК',
              })}
              text={i18n('Чтобы не вводить при доставке', {
                context: 'Описание шортката адресов в блоке заполнения профиля в ЛК',
              })}
              label={i18n('Адреса', {
                context: 'Заголовок шортката адресов в блоке заполнения профиля в ЛК',
              })}
              labelBold
              withImage
              variant="pin"
              tall
              color="info"
              {...logProps('shortcut-link.add-address')}
            />
          </CarouselItem>
        )}
        {hasAvatarAndFirstname && (
          <CarouselItem>
            <Shortcut
              size="s"
              buttonText={i18n('Заполнено', {
                context: 'Текст задизейбленной кнопки в шорткате заполнения профиля в ЛК',
              })}
              text={i18n('Как к вам обращаться?', {
                context: 'Описание шортката ФИО и аватара в блоке заполнения профиля в ЛК',
              })}
              label={i18n('Имя и фото', {
                context: 'Заголовок шортката ФИО и аватара в блоке заполнения профиля в ЛК',
              })}
              labelBold
              withImage
              variant="camera"
              tall
              color="success"
              disabled
            />
          </CarouselItem>
        )}
        {hasBirthdate && (
          <CarouselItem>
            <Shortcut
              size="s"
              buttonText={i18n('Заполнено', {
                context: 'Текст задизейбленной кнопки в шорткате заполнения профиля в ЛК',
              })}
              text={i18n('Чтобы вовремя поздравить', {
                context: 'Описание шортката дня рождения в блоке заполнения профиля в ЛК',
              })}
              label={i18n('День рождения', {
                context: 'Заголовок шортката дня рождения в блоке заполнения профиля в ЛК',
              })}
              labelBold
              withImage
              variant="salute"
              tall
              color="success"
              disabled
            />
          </CarouselItem>
        )}
        {hasPhone && (
          <CarouselItem>
            <Shortcut
              size="s"
              buttonText={i18n('Заполнено', {
                context: 'Текст задизейбленной кнопки в шорткате заполнения профиля в ЛК',
              })}
              text={i18n('Поможет усилить защиту данных', {
                context: 'Описание шортката телефона в блоке заполнения профиля в ЛК',
              })}
              label={i18n('Телефон', {
                context: 'Заголовок шортката телефона в блоке заполнения профиля в ЛК',
              })}
              labelBold
              withImage
              variant="phone"
              tall
              color="success"
              disabled
            />
          </CarouselItem>
        )}
        {hasAddresses && (
          <CarouselItem>
            <Shortcut
              size="s"
              buttonText={i18n('Заполнено', {
                context: 'Текст задизейбленной кнопки в шорткате заполнения профиля в ЛК',
              })}
              text={i18n('Чтобы не вводить при доставке', {
                context: 'Описание шортката адресов в блоке заполнения профиля в ЛК',
              })}
              label={i18n('Адреса', {
                context: 'Заголовок шортката адресов в блоке заполнения профиля в ЛК',
              })}
              labelBold
              withImage
              variant="pin"
              tall
              color="success"
              disabled
            />
          </CarouselItem>
        )}
      </Carousel>
    </Section>
  );
};
