import { useMemo } from 'react';

import { useIdPaths, usePassportPaths } from '@client/shared/paths';

import { RecoveryMethodsData } from '../api/store';
import { i18n } from './i18n';

type DataKeys = keyof RecoveryMethodsData;

const hints = {
  success: () => i18n('Вы подключили все способы восстановления'),
  phone: () => i18n('Добавьте телефон, чтобы защитить аккаунт'),
  question: () => i18n('Добавьте контрольный вопрос, чтобы усилить защиту'),
  emails: () => i18n('Добавьте почту, чтобы усилить защиту'),
};

const priority: DataKeys[] = ['phone', 'question', 'emails'];

export function useSecurityProgress(data: RecoveryMethodsData | null) {
  const passportPaths = usePassportPaths();
  const idPaths = useIdPaths();

  return useMemo(() => {
    const links: Record<DataKeys, string> = {
      phone: passportPaths.phones,
      question: idPaths.securityControlQuestion,
      emails: passportPaths.emailAdd,
    };

    if (data === null) {
      return {
        total: 0,
        progress: 0,
        hint: '',
      };
    }

    let progress = 0;
    let hintKey: DataKeys | 'success' = 'success';

    for (const key of priority) {
      const value = data[key];
      const isValid = Array.isArray(value) ? value.length > 0 : Boolean(value);

      if (isValid) {
        progress++;
      } else if (hintKey === 'success') {
        hintKey = key;
      }
    }

    return {
      total: priority.length,
      progress,
      hint: hints[hintKey](),
      link: hintKey !== 'success' ? links[hintKey] : undefined,
    };
  }, [data, idPaths, passportPaths]);
}
