/* eslint-disable @yandex-id/i18n/no-unwrapped-strings */
import cx from 'classnames';
import { CSSProperties, VFC, useCallback, useMemo } from 'react';

import { addressModel } from '@client/features/address-editor';
import { AddressType } from '@client/shared/api/graphql';
import { Text } from '@client/shared/ui-kit';
import { mergeProps, useHover, usePress } from '@use-platform/react';

import type { Address as AddressCardProps } from '../../api';
import { AddHome } from './assets/AddHome';
import { AddOther } from './assets/AddOther';
import { AddWork } from './assets/AddWork';
import { Home } from './assets/Home';
import { Other } from './assets/Other';
import { Work } from './assets/Work';
import fallbackMapSvg from './assets/fallback-map.svg';

import styles from './AddressCard.module.css';

export const AddressCard: VFC<AddressCardProps> = (props) => {
  const { type, previewUrl = fallbackMapSvg.src, shortText, id } = props;

  const isFilled = Boolean(id);

  const label = getLabel(type);
  const fallbackText = getFallbackText(type);
  const IconType = getIconType(type);
  const AddIconType = getAddIconType(type);

  const onPress = useCallback(() => {
    addressModel.addressPressed(props);
  }, [props]);

  const { pressProps } = usePress({ onPress });
  const { isHovered, hoverProps } = useHover({});

  const style = useMemo<CSSProperties>(
    () => ({ backgroundImage: `url("${previewUrl}")` }),
    [previewUrl],
  );

  return (
    <div
      {...mergeProps(hoverProps, pressProps)}
      className={cx(styles.root, { [styles.root_isHovered]: isHovered })}
      style={style}
      role="button"
      tabIndex={0}
    >
      <div className={styles.pointer}>{isFilled ? <IconType /> : <AddIconType />}</div>

      <div className={styles.content}>
        {isFilled && (
          <Text color="secondary" variant="text-xs">
            {label}
          </Text>
        )}
        <Text
          color="primary"
          maxLines={3}
          overflow="ellipsis"
          variant="text-s"
          weight={isFilled ? 'regular' : 'medium'}
        >
          {shortText ?? fallbackText}
        </Text>
      </div>
    </div>
  );
};

function getLabel(type: AddressType) {
  return {
    [AddressType.Home]: 'Дом',
    [AddressType.Work]: 'Работа',
    [AddressType.Other]: 'Другой',
  }[type];
}

function getFallbackText(type: AddressType) {
  return {
    [AddressType.Home]: 'Добавить домашний адрес',
    [AddressType.Work]: 'Добавить рабочий адрес',
    [AddressType.Other]: 'Добавить ещё один адрес',
  }[type];
}

function getIconType(type: AddressType) {
  return {
    [AddressType.Home]: Home,
    [AddressType.Work]: Work,
    [AddressType.Other]: Other,
  }[type];
}

function getAddIconType(type: AddressType) {
  return {
    [AddressType.Home]: AddHome,
    [AddressType.Work]: AddWork,
    [AddressType.Other]: AddOther,
  }[type];
}
