import RootApp from 'next/app';
import { ReactElement, StrictMode } from 'react';

import '@client/application/global.css';
import { Head } from '@client/application/head';
import { ThemeProvider } from '@client/shared/design-system';
import { ApolloProvider, useApolloClient } from '@client/shared/libs/apollo';
import { canUseDom } from '@client/shared/libs/can-use-dom';
import { EnvProvider } from '@client/shared/libs/env';
import { LANG, getI18nLang, setI18nLang } from '@client/shared/libs/i18n';
import { useMetrikaInit } from '@client/shared/libs/metrika';
import { withHydrate } from '@client/shared/libs/mobx-next';
import { snackbarStore } from '@client/shared/libs/notifications';
import '@client/shared/libs/rum/bundle';
import { getServerSideProps } from '@client/shared/next';
import { PageMainLayout } from '@client/shared/page-layouts';
import { SnackbarContainer } from '@client/shared/ui-kit';
import { NextApp, NextAppInitialProps } from '@shared/types/next-bridge';
import { LocaleProvider, SSRProvider } from '@use-platform/react';

const App: NextApp = (props) => {
  const { Component, pageProps, serverSideProps, apolloState } = props;
  const apolloClient = useApolloClient({ initialState: apolloState });
  const getLayout = Component.getLayout ?? defaultLayout;

  // Устанавливаем локаль для SSR
  if (!canUseDom) {
    const locale = props.router.locale as LANG | undefined;

    locale && setI18nLang(locale);
  }

  useMetrikaInit(serverSideProps.yenv);

  return (
    <EnvProvider value={serverSideProps}>
      <LocaleProvider locale={getI18nLang()}>
        <ApolloProvider client={apolloClient}>
          <SSRProvider>
            <ThemeProvider>
              <Head />
              {getLayout(
                <StrictMode>
                  <Component {...pageProps} />
                </StrictMode>,
              )}
              <SnackbarContainer store={snackbarStore} position="bottom-center" maxCount={1} />
            </ThemeProvider>
          </SSRProvider>
        </ApolloProvider>
      </LocaleProvider>
    </EnvProvider>
  );
};

App.getInitialProps = async (context) => {
  const initialProps = (await RootApp.getInitialProps(context)) as NextAppInitialProps;
  const serverSideProps = getServerSideProps();

  if (serverSideProps) {
    return { ...initialProps, serverSideProps };
  }

  return initialProps;
};

function defaultLayout(page: ReactElement) {
  return <PageMainLayout>{page}</PageMainLayout>;
}

export default withHydrate(App);
