import fs from 'fs';
import path from 'path';

import Document, { DocumentProps, Head, Html, Main, NextScript } from 'next/document';
import { ComponentType } from 'react';

import {
  ApolloClient,
  ApolloSsrProvider,
  NormalizedCacheObject,
  createSchemaClient,
  getDataFromTree,
} from '@client/shared/libs/apollo/ssr';
import { ButterflyScript } from '@client/shared/libs/butterfly';
import { LANG } from '@client/shared/libs/i18n';
import { RumInline } from '@client/shared/libs/rum/inline-script';
import { renderPageWithEnhance } from '@client/shared/next';
import { getYEnv } from '@server/config/env-config';
import { staticLogger } from '@server/shared/logger';
import { DocumentContext, ServerSideProps } from '@shared/types/next-bridge';

type DocumentFiles = {
  sharedFiles: readonly string[];
  pageFiles: readonly string[];
  allFiles: readonly string[];
};

type Page = string;
type ChunkName = string;
type ID = string;
let LocaleManifest: Record<Page, Record<LANG, Record<ChunkName, ID>>>;

function loadLocaleManifest() {
  if (getYEnv() === 'development') {
    return;
  }

  try {
    LocaleManifest = JSON.parse(
      fs.readFileSync(path.join(__dirname, '../../', 'locales-manifest.json'), 'utf8'),
    );
  } catch (err) {
    staticLogger.error('Failed to load locales manifest', { err });
  }
}

loadLocaleManifest();

const rumInline = new RumInline(
  path.join(__dirname, '../../', 'inline-rum-bundle.js'),
  staticLogger,
);

class IdHead extends Head {
  getScripts(files: DocumentFiles): JSX.Element[] {
    const data = this.context.__NEXT_DATA__;
    const lang = data.locale as LANG;
    // отрезаем первый /
    const page = data.page.substring(1);

    // Находим в манифесте языковые бандлы для данного языка
    // чтобы подложить их на страницу заранее
    const chunkList = Object.keys(LocaleManifest?.[page]?.[lang] || {});
    let newFiles = files;

    if (chunkList?.length) {
      newFiles = {
        ...files,
        allFiles: files.allFiles.concat(chunkList),
        pageFiles: files.pageFiles.concat(chunkList),
      };
    }

    return super.getScripts(newFiles);
  }
}

interface EnhancerProps {
  apolloClient: ApolloClient;
  serverSideProps: ServerSideProps;
}

function createEnhancer(enhancerProps: EnhancerProps) {
  const { apolloClient, ...otherProps } = enhancerProps;

  return <T extends {}>(App: ComponentType<T>): ComponentType<T> => {
    return (props) => {
      return (
        <ApolloSsrProvider client={apolloClient}>
          <App {...props} {...otherProps} />
        </ApolloSsrProvider>
      );
    };
  };
}

interface EnhancedDocumentInitialProps {
  nonce?: string;
  apolloState?: NormalizedCacheObject;
  serverSideProps: ServerSideProps;
}

type EnhancedDocumentProps = DocumentProps & EnhancedDocumentInitialProps;

class EnhancedDocument extends Document<EnhancedDocumentInitialProps> {
  static async getInitialProps(ctx: DocumentContext) {
    const { req, renderPage } = ctx;
    const nextBridge = req?.nextBridge;

    if (!nextBridge) {
      // HACK: При старте nextjs запрашивать webpack-hmr и вызывает функцию без контекста,
      // поэтому возвращаем пустой документ для того, чтобы не было ошибок на странице.
      return { html: '', head: [], styles: [] };
    }

    const serverSideProps = await nextBridge.getServerSideProps(ctx);
    const apolloClient = createSchemaClient({ schema: nextBridge.graphqlSchema });
    const nonce = nextBridge.getNonce();
    const enhanceApp = createEnhancer({ apolloClient, serverSideProps });

    ctx.renderPage = renderPageWithEnhance(enhanceApp, renderPage);

    const AppTree = enhanceApp(ctx.AppTree);
    const apolloState = await getDataFromTree(apolloClient, <AppTree pageProps={{}} />);

    const initialProps = await Document.getInitialProps(ctx);

    return { ...initialProps, serverSideProps, nonce, apolloState };
  }

  constructor(props: EnhancedDocumentProps) {
    super(props);

    props.__NEXT_DATA__.props.apolloState = props.apolloState;
    props.__NEXT_DATA__.props.serverSideProps = props.serverSideProps;
  }

  render() {
    const { nonce, serverSideProps } = this.props;

    return (
      <Html>
        <IdHead nonce={nonce}>
          {rumInline.getInlineScript(serverSideProps, nonce, this.props.__NEXT_DATA__.locale)}
        </IdHead>
        <body data-theme="default" data-color="light">
          <Main />
          <NextScript nonce={nonce} />
          {serverSideProps?.isYandexoid && <ButterflyScript />}
        </body>
      </Html>
    );
  }
}

export default EnhancedDocument;
