/* eslint-disable @yandex-id/i18n/no-unwrapped-strings */
import cx from 'classnames';
import { VFC, useRef, useState } from 'react';

import { useDevice } from '@client/shared/hooks';
import { PayServiceIcon } from '@client/shared/icons';
import { logProps } from '@client/shared/libs/metrika';
import { RouterLink } from '@client/shared/libs/router-link';
import { ImageIcon } from '@client/shared/ui-kit/ImageIcon';
import { mergeProps, useHover, usePress } from '@use-platform/react';
import { useIsomorphicLayoutEffect } from '@use-platform/react/libs/isomorphic-layout-effect';

import styles from './Card.module.css';

export type CardColor = 'orange' | 'blue' | 'yellow' | 'violet' | 'red';
export type CardBadgeType = 'pay' | 'active';

export interface CardProps {
  href: string;
  label: string;
  color: CardColor;
  badge?: CardBadgeType;
  cover?: StaticImageData | string;
  logEvent?: string;
}

const SIZE_THRESHOLD = 10;

export const Card: VFC<CardProps> = (props) => {
  const { href, label, color, badge, cover, logEvent } = props;

  const containerRef = useRef<HTMLAnchorElement>(null);
  const [isWide, setWide] = useState(false);
  const { isTouch } = useDevice();
  const { hoverProps, isHovered } = useHover({
    // TODO: Удалить после https://github.com/use-platform/use-platform/issues/140
    disabled: isTouch,
  });
  const { pressProps, isPressed } = usePress({});

  useIsomorphicLayoutEffect(() => {
    function onWindowResize() {
      if (!containerRef.current) {
        return;
      }

      const { clientWidth, clientHeight } = containerRef.current;
      const diff = clientWidth - clientHeight;

      setWide(diff > SIZE_THRESHOLD);
    }

    onWindowResize();

    window.addEventListener('resize', onWindowResize);

    return () => {
      window.removeEventListener('resize', onWindowResize);
    };
  }, []);

  const isExternalUrl = href.match(/^http/);

  return (
    <RouterLink
      innerRef={containerRef}
      href={href}
      target={isExternalUrl ? '_blank' : undefined}
      className={cx(
        styles.root,
        {
          [styles.root_isHovered]: isHovered,
          [styles.root_isPressed]: isPressed,
          [styles.root_isWide]: isWide,
        },
        styles[`root_color-${color}`],
      )}
      {...mergeProps(hoverProps, pressProps, logEvent ? logProps(logEvent) : {})}
    >
      <div className={styles.label}>{label}</div>

      {badge && (
        <div className={styles.badge}>
          {badge === 'pay' && <PayServiceIcon className={styles.payBadge} />}
          {badge === 'active' && <span className={styles.textBadge}>активна</span>}
        </div>
      )}

      {cover && (
        <span className={styles.cover}>
          <ImageIcon src={cover} size={340} />
        </span>
      )}
    </RouterLink>
  );
};
