import qs from 'querystring';

import { Injectable } from '@nestjs/common';
import {
  AddRecoveryEmailInput,
  AddRecoveryEmailPayload,
  AddRecoveryEmailProblem,
  AddRecoveryEmailProblemKind,
  ConfirmRecoveryEmailByCodeInput,
  ConfirmRecoveryEmailByCodePayload,
  ConfirmRecoveryEmailByCodeProblem,
  ConfirmRecoveryEmailByCodeProblemKind,
  ConfirmRecoveryEmailByLinkInput,
  ConfirmRecoveryEmailByLinkPayload,
  ConfirmRecoveryEmailByLinkProblem,
  ConfirmRecoveryEmailByLinkProblemKind,
  RemoveRecoveryEmailInput,
  RemoveRecoveryEmailPayload,
  RemoveRecoveryEmailProblem,
  RemoveRecoveryEmailProblemKind,
} from '@server/graphql-schema';
import { fromModel } from '@server/shared/libs';
import { LoggerService } from '@server/shared/logger';
import { PassportApiResponse, PassportService } from '@server/shared/passport';
import { Context } from '@yandex-int/nest-common';

@Injectable()
export class EmailService {
  constructor(
    private context: Context,
    private logger: LoggerService,
    private passport: PassportService,
  ) {}

  async addRecoveryEmail(input: AddRecoveryEmailInput) {
    const language = this.context.req.langdetect.id;

    if (!input.trackId) {
      input.trackId = await this.passport.getTrackId();
    }

    const payload: Record<string, string> = {
      email: input.email,
      language,
      retpath: input.redirectUrl,
      track_id: input.trackId,
      validator_ui_url: input.validationUrl,
    };

    const response = await this.post('/1/bundle/email/send_confirmation_email/', payload);

    if (response.data.status === 'error') {
      const { errors } = response.data;
      let reason = AddRecoveryEmailProblemKind.INTERNAL;

      if (errors.includes('password.required')) {
        reason = AddRecoveryEmailProblemKind.PASSWORD_REQUIRED;
      } else if (errors.includes('email.already_sent')) {
        reason = AddRecoveryEmailProblemKind.TOO_MANY_REQUESTS;
      } else if (errors.includes('email.is_native')) {
        reason = AddRecoveryEmailProblemKind.IS_NATIVE;
      } else if (errors.includes('email.already_confirmed')) {
        reason = AddRecoveryEmailProblemKind.ALREADY_CONFIRMED;
      } else if (errors.includes('email.invalid')) {
        reason = AddRecoveryEmailProblemKind.EMAIL_INVALID;
      } else {
        this.logger.error('AddRecoveryEmailError', errors);
      }

      return fromModel(AddRecoveryEmailProblem, { trackId: input.trackId, reason });
    }

    return fromModel(AddRecoveryEmailPayload, { trackId: input.trackId });
  }

  async removeRecoveryEmail(input: RemoveRecoveryEmailInput) {
    if (!input.trackId) {
      input.trackId = await this.passport.getTrackId();
    }

    const payload: Record<string, string> = {
      email: input.email,
    };

    const response = await this.post('/1/bundle/email/delete/', payload);

    if (response.data.status === 'error') {
      const { errors } = response.data;
      let reason = RemoveRecoveryEmailProblemKind.INTERNAL;

      if (errors.includes('password.required')) {
        reason = RemoveRecoveryEmailProblemKind.PASSWORD_REQUIRED;
      } else {
        this.logger.error('RemoveRecoveryEmailError', errors);
      }

      return fromModel(RemoveRecoveryEmailProblem, { trackId: input.trackId, reason });
    }

    return fromModel(RemoveRecoveryEmailPayload, {});
  }

  async confirmEmailByLink(input: ConfirmRecoveryEmailByLinkInput) {
    const payload: Record<string, string> = {
      key: input.code,
    };

    const response = await this.post('/1/bundle/email/confirm/by_link/', payload);

    if (response.data.status === 'error') {
      const { errors } = response.data;
      let reason = ConfirmRecoveryEmailByLinkProblemKind.INTERNAL;

      if (errors.includes('email.incorrect_key')) {
        reason = ConfirmRecoveryEmailByLinkProblemKind.INCORRECT_KEY;
      } else if (errors.includes('email.already_confirmed')) {
        reason = ConfirmRecoveryEmailByLinkProblemKind.ALREADY_CONFIRMED;
      } else {
        this.logger.error('ConfirmRecoveryEmailByLinkError', errors);
      }

      return fromModel(ConfirmRecoveryEmailByLinkProblem, { reason });
    }

    return fromModel(ConfirmRecoveryEmailByLinkPayload, {});
  }

  async confirmEmailByCode(input: ConfirmRecoveryEmailByCodeInput) {
    const payload: Record<string, string> = {
      key: input.code,
      track_id: input.trackId,
    };

    const response = await this.post('/1/bundle/email/confirm/by_code/', payload);

    if (response.data.status === 'error') {
      const { errors } = response.data;
      let reason = ConfirmRecoveryEmailByCodeProblemKind.INTERNAL;

      if (errors.includes('email.incorrect_key')) {
        reason = ConfirmRecoveryEmailByCodeProblemKind.INCORRECT_KEY;
      } else if (errors.includes('email.already_confirmed')) {
        reason = ConfirmRecoveryEmailByCodeProblemKind.ALREADY_CONFIRMED;
      } else if (errors.includes('email.key_check_limit_exceeded')) {
        reason = ConfirmRecoveryEmailByCodeProblemKind.KEY_CHECK_LIMIT_EXCEEDED;
      } else {
        this.logger.error('ConfirmRecoveryEmailByCodeError', errors);
      }

      return fromModel(ConfirmRecoveryEmailByCodeProblem, { reason });
    }

    return fromModel(ConfirmRecoveryEmailByCodePayload, {});
  }

  private post<T, U extends Record<string, any> = any>(path: string, data: U) {
    return this.passport.http.post<PassportApiResponse<T>>(
      `${path}?consumer=passport`,
      qs.stringify(data),
      {
        headers: {
          'Ya-Client-Cookie': this.context.req.headers.cookie,
          'Ya-Client-Host': this.context.req.headers.host,
        },
      },
    );
  }
}
