import { Injectable } from '@nestjs/common';
import {
  ExternalAccount,
  ExternalAccountProvider,
  ExternalAccountProviderName,
  ToggleLoginWithExternalAccountInput,
  ToggleLoginWithExternalAccountProblem,
  ToggleLoginWithExternalAccountReason,
  UnbindExternalAccountInput,
  UnbindExternalAccountProblem,
  UnbindExternalAccountReason,
} from '@server/graphql-schema';
import { fromModel } from '@server/shared/libs';
import { PassportService } from '@server/shared/passport';
import { SocialProfile } from '@server/shared/passport/api-types';

const PROVIDERS = ['vk', 'fb', 'tw', 'mr', 'gg', 'ok', 'esia'];

@Injectable()
export class ExternalAccountService {
  constructor(private passport: PassportService) {}

  async getAccounts() {
    const response = await this.passport.getSocialProfiles();

    if (response.status === 'error') {
      // TODO: Тут по хорошему нужно отдавать ошибку
      return [];
    }

    return response.profiles
      .filter((profile) => PROVIDERS.includes(profile.provider_code) && !profile.phonish)
      .map((profile) => {
        const {
          profile_id: profileId,
          allow_auth: allowAuth,
          provider_code: providerCode,
          person,
        } = profile;
        const { firstname: firstName = null, lastname: lastName = null } = person || {};
        const id = profileId.toString();

        const provider = fromModel(ExternalAccountProvider, {
          id: providerCode,
          name: this.getProviderName(providerCode),
        });

        return fromModel(ExternalAccount, {
          id,
          allowAuth,
          provider,
          firstName,
          lastName,
          displayName: this.getDisplayName(profile),
        });
      });
  }

  getAccountProviders() {
    return PROVIDERS.map((id) => {
      return fromModel(ExternalAccountProvider, {
        id,
        name: this.getProviderName(id),
      });
    });
  }

  async unbindAccount(_input: UnbindExternalAccountInput) {
    // TODO: implement logic
    return fromModel(UnbindExternalAccountProblem, {
      reason: UnbindExternalAccountReason.INTERNAL,
    });
  }

  async toggleLoginWithAccount(_input: ToggleLoginWithExternalAccountInput) {
    // TODO: implement logic
    return fromModel(ToggleLoginWithExternalAccountProblem, {
      reason: ToggleLoginWithExternalAccountReason.INTERNAL,
    });
  }

  private getProviderName(id: string) {
    switch (id) {
      case 'apl':
        return ExternalAccountProviderName.APPLE;

      case 'esia':
        return ExternalAccountProviderName.ESIA;

      case 'fb':
        return ExternalAccountProviderName.FACEBOOK;

      case 'gg':
        return ExternalAccountProviderName.GOOGLE;

      case 'mr':
        return ExternalAccountProviderName.MAILRU;

      case 'ok':
        return ExternalAccountProviderName.ODNOKLASSNIKI;

      case 'tw':
        return ExternalAccountProviderName.TWITTER;

      case 'vk':
        return ExternalAccountProviderName.VK;
    }

    throw new Error(`Unknown provider id: ${id}`);
  }

  private getDisplayName(profile: SocialProfile) {
    const { person, username, userid } = profile;
    const { firstname, lastname } = person || {};

    if (firstname && lastname) {
      return `${firstname} ${lastname}`.trim();
    }

    if (username) {
      return username;
    }

    return userid;
  }
}
