import { ApolloError } from 'apollo-server-express';

import { Injectable } from '@nestjs/common';
import { BlackboxApiService } from '@server/features/blackbox-api';
import { Family } from '@server/graphql-schema';
import { fromModel } from '@server/shared/libs';
import { LoggerService } from '@server/shared/logger';
import { PassportService } from '@server/shared/passport';
import { FamilyPayService } from '@server/sources/familypay';

const CENTS = 100;

class FamilyDataException extends Error {
  readonly errors: Object[];

  constructor(message: string, errors: Object[]) {
    super(message);

    this.errors = errors;
  }
}

@Injectable()
export class FamilyService {
  constructor(
    private logger: LoggerService,
    private passport: PassportService,
    private blackbox: BlackboxApiService,
    private familypay: FamilyPayService,
  ) {}

  async getFamily() {
    try {
      const response = await this.passport.getAccountWithFamilyData();

      if (response.data.status === 'error') {
        throw new FamilyDataException('Passport request failed', response.data.errors);
      }

      const {
        account: { family_info: info },
        family_members: members,
        family_invites: invites,
        family_kids: kids,
        family_settings: settings,
      } = response.data;

      let securePhones: Record<string, boolean> = {};
      let familyPayData;

      if (members) {
        try {
          securePhones = await this.blackbox.checkSecurePhones(
            members.map((member) => String(member.uid)),
          );
        } catch (error) {
          this.logger.error('[FamilyApi] getSecurePhones error', { error });
        }
      }

      if (info?.family_id) {
        try {
          familyPayData = await this.familypay.getFamilyPayInfo(info.family_id);
        } catch (error) {
          this.logger.error('[FamilyApi] getFamilyPayData error', { error });
        }
      }

      const adminUid = info?.admin_uid;

      return fromModel(Family, {
        info: info
          ? {
              id: info.family_id,
              adminUid: String(adminUid),
            }
          : undefined,

        members:
          members?.map((member) => ({
            avatar: member.default_avatar,
            name: member.display_name,
            hasPlus: member.has_plus,
            uid: String(member.uid),
            isAdmin: member.uid === adminUid,
            placeId: member.place_id,
            hasSecurePhone: securePhones[member.uid],
          })) || [],

        invites:
          invites?.map((invite) => ({
            id: invite.invite_id,
            contact: invite.contact,
            sendMethod: invite.send_method,
          })) || [],

        kids:
          kids?.map((kid) => ({
            uid: String(kid.uid),
            name: kid.display_name?.name,
            avatar: kid.display_name?.default_avatar,
            gender: kid.person?.gender,
            birthday: kid.person?.birthday,
            contentRating: kid.content_rating_class,
            musicRating: kid.music_content_rating_class,
            videoRating: kid.video_content_rating_class,
            placeId: kid.place_id,
          })) || [],

        settings: settings
          ? {
              maxCapacity: settings.max_capacity,
              maxKids: settings.max_kids_number,
            }
          : undefined,

        pay: familyPayData?.data
          ? {
              cardInfo: familyPayData.data.properties.cardInfo,
              userRestrictions: familyPayData.data.userRestrictions.map((restriction) => ({
                ...restriction,
                uid: String(restriction.uid),
                balance: (restriction.limit.value - restriction.expenses.amount) / CENTS,
                limit: {
                  value: restriction.limit.value / CENTS,
                  limitMode: restriction.limit.limitMode,
                },
                hasSecurePhone: securePhones[restriction.uid],
              })),
            }
          : undefined,
      });
    } catch (err) {
      this.logger.error('[FamilyApi] getFamily error', { err });

      throw new ApolloError('Failed to get family data', 'FAMILY_DATA_ERROR');
    }
  }
}
