import { createHash } from 'crypto';

import { Injectable, InternalServerErrorException } from '@nestjs/common';
import { FuturePayment, FuturePaymentType } from '@server/graphql-schema';
import { BlackboxService } from '@server/shared/blackbox';
import { SplitService } from '@server/sources/split';

@Injectable()
export class FuturePaymentsService {
  constructor(private blackboxService: BlackboxService, private splitService: SplitService) {}

  async getPayments(): Promise<FuturePayment[]> {
    try {
      const blackbox = await this.blackboxService.getBlackbox();
      const splitUserStat = await this.splitService.getUserStatByUid(blackbox.uid ?? '');

      if (!splitUserStat.account_stat.next_payment_date) {
        return [];
      }

      return [
        {
          id: createHash('sha256')
            .update(
              `market${FuturePaymentType.SPLIT}${splitUserStat.account_stat.next_payment_date}${splitUserStat.account_stat.next_payment_amount}`,
            )
            .digest('hex'),
          service: 'market',
          type: FuturePaymentType.SPLIT,
          nextPaymentAt: splitUserStat.account_stat.next_payment_date,
          amount: Number(splitUserStat.account_stat.next_payment_amount),
          iframeUrl: splitUserStat.account_url,
        },
      ];
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }
}
