import { DynamicModule, Module } from '@nestjs/common';
import { APP_FILTER, APP_INTERCEPTOR } from '@nestjs/core';
import { IpregModule } from '@yandex-id/ipreg';
import { AsyncStorageModule, RequestIdModule } from '@yandex-int/nest-common';
import {
  SecretkeyModule,
  TvmModule,
  UatraitsModule,
  YandexuidModule,
} from '@yandex-int/nest-infra';

import { EmailModule } from './api/email';
import { ExternalAccountModule } from './api/external-account';
import { FamilyModule } from './api/family';
import { FuturePaymentsModule } from './api/future-payments';
import {
  BlackboxConfigService,
  CSPConfigService,
  ConfigModule,
  IpregConfigService,
  TvmConfigService,
  isWatchMode,
} from './config';
import { SECRET_KEY_SALT } from './config/constants';
import { AccountsModule } from './features/accounts';
import { AddressModule } from './features/address';
import { AvatarsProxyModule } from './features/avatars-proxy';
import { BillsModule } from './features/bills';
import { ConfirmationModule } from './features/confirmation';
import { DiskModule } from './features/disk';
import { DocumentsModule } from './features/documents';
import { LocationModule } from './features/location';
import { OhioModule } from './features/ohio';
import { SearchModule } from './features/search';
import { ViewerModule } from './features/viewer';
import { InfrastructureModule } from './infrastructure';
import { PagesModule } from './pages';
import { BlackboxModule } from './shared/blackbox';
import { CSPModule } from './shared/csp';
import { ErrorsLogInterceptor } from './shared/errors/errors.interceptor';
import { ExperimentsModule } from './shared/experiments';
import { LangDetectModule } from './shared/lang-detect';
import { LocaleInterceptor } from './shared/locale/locale.interceptor';
import { LoggerModule } from './shared/logger';
import { MiddlewaresModule } from './shared/middlewares';
import { NextModule } from './shared/next';
import { NextBridgeModule } from './shared/next-bridge';

@Module({})
export class AppModule {
  static async register(): Promise<DynamicModule> {
    return {
      module: AppModule,
      providers: [
        { provide: APP_INTERCEPTOR, useClass: ErrorsLogInterceptor },
        { provide: APP_FILTER, useClass: ErrorsLogInterceptor },
        { provide: APP_INTERCEPTOR, useClass: LocaleInterceptor },
      ],
      imports: [
        ConfigModule.forRoot(),

        // infrastructure
        AsyncStorageModule.forRoot(),
        RequestIdModule.forRoot({
          uid: () => `${Date.now()}-yid-${Math.random().toFixed(12).slice(2)}`,
          headerName: 'x-request-id',
        }),
        IpregModule.forRootAsync({ useExisting: IpregConfigService }),
        UatraitsModule.forRoot(),
        SecretkeyModule.forRoot({ version: 2, salt: SECRET_KEY_SALT }),
        LoggerModule,

        TvmModule.forRootAsync({
          inject: [TvmConfigService],
          useFactory: (config: TvmConfigService) => config.options,
        }),
        BlackboxModule.forRootAsync({
          inject: [BlackboxConfigService],
          useFactory: (config: BlackboxConfigService) => config.options,
        }),
        LangDetectModule,
        YandexuidModule.forRoot(),
        MiddlewaresModule,
        CSPModule.forRootAsync({ useExisting: CSPConfigService }),
        ExperimentsModule,
        InfrastructureModule,

        // features
        AddressModule,
        AvatarsProxyModule,
        ConfirmationModule,
        DiskModule,
        DocumentsModule,
        FuturePaymentsModule,
        SearchModule,
        ViewerModule,
        AccountsModule,
        ExternalAccountModule,
        LocationModule,
        EmailModule,
        OhioModule,
        FamilyModule,
        BillsModule,

        // pages
        NextBridgeModule,
        NextModule.forRootAsync({
          serverOptions: {
            dev: isWatchMode,
            isNextDevCommand: isWatchMode,
          },
        }),
        PagesModule,
      ],
    };
  }
}
