import { ExtensionConfig, LusterProcess, MasterProcess, WorkerProcess } from 'luster';

import { staticLogger } from '@server/shared/logger';

import { statCountEvent } from '../../shared/stats';

export interface ClusterStatusConfig {
  processNamePrefix: string;
  /** Какое количество воркеров считать порогом успешного запуска службы */
  readyThreshold: number;
  /** Интервал опроса воркеров (мс) */
  workerPingInterval: number;
  /**
   * Таймаут для рестарта неответивших после пинга воркеров (мс)
   * Осмысленно делать меньше workerPingInterval
   * */
  workerTimeout: number;
  ignoreException: boolean;
}

export interface PongerData {
  seq: number;
  timeout: NodeJS.Timeout;
}

module.exports = {
  configure: function (
    config: ExtensionConfig<ClusterStatusConfig>,
    currentProcess: LusterProcess,
  ) {
    if (currentProcess.isMaster) {
      // eslint-disable-next-line @typescript-eslint/no-use-before-define
      return new Master(config, currentProcess);
    }

    if (currentProcess.isWorker) {
      // eslint-disable-next-line @typescript-eslint/no-use-before-define
      return new Worker(config, currentProcess);
    }
  },
};

class Master {
  workerStarting = true;

  // При закрытии процесса нам не надо записывать ошибки от воркеров
  ignoreWorkerErrors = false;
  // В некоторых случаях мы не хотим убивать процесс сами, а ждем рестарта от ватчдога
  waitsRestart = false;
  // Выключаемся
  shuttingDown = false;

  constructor(private config: ExtensionConfig<ClusterStatusConfig>, private master: MasterProcess) {
    this.handleErrors();
    this.handleSignals();
    this.setProcessTitle();

    this.startInnerPingProcess();

    this.handleWorkersDead();
  }

  handleErrors() {
    process.once('uncaughtException', (err) => {
      statCountEvent('master_uncaught_exception');
      staticLogger.error('UNCAUGHT_EXCEPTION', {
        message: err.message || err,
        stack: err.stack,
      });

      // eslint-disable-next-line no-process-exit
      process.exit(1);
    });

    // Непонятно когда мастер может вызвать это событие, но если вызовет нам лучше уйти в рестарт
    this.master.on('error', (err) => {
      statCountEvent('master_cluster_error');
      staticLogger.error('CLUSTER_ERROR', {
        message: err.message || err,
      });
      this.waitsRestart = true;
    });
  }

  handleSignals() {
    const stopOn = (signal: NodeJS.Signals) => {
      process.once(signal, () => {
        staticLogger.log(signal);
        this.shuttingDown = true;
        this.ignoreWorkerErrors = true;
        this.master.shutdown();
      });
    };

    stopOn('SIGQUIT');
    stopOn('SIGINT');

    process.on('SIGUSR2', () => {
      staticLogger.log('SIGUSR2');
      this.master.softRestart();
    });
  }

  /**
   * В списке процессов все будет выглядеть именно так
   */
  setProcessTitle() {
    process.title = this.config.get('processNamePrefix') + '-master';
  }

  /**
   * Раз в секунду опрашиваем воркеры во IPC протоколу.
   * Если воркер не ответил - рестартуем его.
   */
  startInnerPingProcess() {
    let seq = 0;
    const pongs: Record<string, PongerData> = {};

    this.master.registerRemoteCommand<{
      request_seq: number;
      wid: string;
    }>('pong', (sender, message) => {
      const ponger = pongs[sender.wid];

      // Если worker не успел ответить за 500мс мы его больше не ждем.
      // Если он все же ответит нам через 1сек мы должны проигнорировать этот ответ.
      // Это очень странная ситуация - worker должен успеть ответить перед тем как мы его убьем.
      if (!ponger || ponger.seq !== message.request_seq) {
        statCountEvent('master_ignored_pong_response');
        staticLogger.error('IGNORED_PONG_RESPONSE', {
          wid: message.wid,
          seq: message.request_seq,
          cur: ponger ? ponger.seq : null,
        });

        return;
      }

      // Если по какой-то причине воркер не знает своего wid нам нужно рестартовать весь кластер
      if (message.wid !== sender.wid || !/^\d+$/.test(message.wid)) {
        this.waitsRestart = true;
        statCountEvent('master_initialization_error');
        staticLogger.error('INITIALIZATION_ERROR', {
          wid: message.wid,
        });

        return;
      }

      clearTimeout(ponger.timeout);
      delete pongs[sender.wid];
    });

    setInterval(() => {
      const workers = this.master.getWorkersArray();

      if (this.workerStarting) {
        const ready = workers.reduce((a, w) => a + (w.ready ? 1 : 0), 0);

        if (ready >= this.config.get('readyThreshold')) {
          this.workerStarting = false;
        }
      }

      workers.forEach((worker) => {
        const wid = worker.wid;

        // Не опрашиваем воркер, если в этот момент он рестартует.
        if (!worker.ready) {
          return;
        }

        // Не даем воркеру второго шанса - если не успел ответить сразу рестартуем
        const timeout = setTimeout(() => {
          delete pongs[wid];
          statCountEvent('master_restart_hanged_worker');
          staticLogger.error('RESTART_HANGED_WORKER', { wid: wid });
          worker.restart();
        }, this.config.get('workerTimeout'));

        pongs[wid] = { seq: seq, timeout: timeout };

        worker.remoteCall('ping', { seq: seq++ });
      });
    }, this.config.get('workerPingInterval')).unref();
  }

  handleWorkersDead() {
    this.master.on('worker exit', (worker) => {
      if (worker.restarting || this.ignoreWorkerErrors) {
        return;
      }

      statCountEvent('master_worker_exit');
      staticLogger.error('WORKER_EXIT', { wid: worker.wid });

      if (!worker.dead) {
        return;
      }

      statCountEvent('master_worker_dead');
      this.waitsRestart = true;
      staticLogger.error('WORKER_DEAD', { wid: worker.wid });
    });
  }
}

class Worker {
  constructor(private config: ExtensionConfig<ClusterStatusConfig>, private worker: WorkerProcess) {
    this.config = config;
    this.worker = worker;

    this.setProcessTitle();

    this.startInnerPongProcess();

    this.handleErrors();
  }

  setProcessTitle() {
    process.title = `${this.config.get('processNamePrefix')}-worker.${this.worker.wid}`;
  }

  startInnerPongProcess() {
    this.worker.registerRemoteCommand<{ seq: number }>('ping', (sender, data) => {
      sender.remoteCall('pong', {
        wid: this.worker.wid,
        request_seq: data.seq,
        state: 'running',
        port: process.env.port,
      });
    });
  }

  handleErrors() {
    process.once('uncaughtException', (err) => {
      statCountEvent('worker_uncaught_exception');
      staticLogger.error('WORKER_UNCAUGHT_EXCEPTION', {
        message: err.message || err,
        stack: err.stack,
      });

      if (this.config.get('ignoreException')) {
        this.handleErrors();

        return;
      }

      // eslint-disable-next-line no-process-exit
      process.exit(1);
    });
  }
}
