import http, { ClientRequest, RequestOptions } from 'http';

import { MasterProcess, WorkerProcess } from 'luster';

import { getTimeDiff, statEvent } from '@server/shared/stats';

interface ErrorWithCode {
  code?: number;
}

module.exports = {
  configure: function (_config: unknown, cluster: WorkerProcess | MasterProcess) {
    if (cluster.isWorker) {
      // @ts-expect-error Утверждается, что addRequest нет в прототипе. Но ТС врёт
      const addRequest = http.Agent.prototype.addRequest;

      // @ts-expect-error аналогично
      http.Agent.prototype.addRequest = function (request, options) {
        handleHttpRequest(request, options);

        return addRequest.call(this, request, options);
      };
    }
  },
};

function getHost(opts: RequestOptions) {
  const proto = (opts.protocol || 'http:') + '//';
  const portValue = opts.port;
  let port = '';

  if (
    portValue &&
    !((proto === 'https://' && portValue === 443) || (proto === 'http://' && portValue === 80))
  ) {
    port = ':' + portValue;
  }

  return proto + String(opts.headers?.host || opts.host || '') + port;
}

interface RequestInfo {
  start: bigint;
  host: string;
  opts: RequestOptions;
}

class RequestsStats {
  private requestsStat = new WeakMap<object, RequestInfo>();

  add(req: object, opts: RequestOptions) {
    const key = {};
    const reqInfo = {
      start: process.hrtime.bigint(),
      host: getHost(opts),
      opts,
    };

    this.requestsStat.set(req, reqInfo);

    statEvent({
      name: 'http.attempt',
      payload: {
        host: reqInfo.host,
        value: 1,
      },
    });

    return key;
  }

  success(req: object, statusCode: number) {
    const { host } = this.requestsStat.get(req) || {};

    statEvent({
      name: 'http.success',
      payload: {
        statusCode,
        host,
        value: this.getTiming(req),
      },
    });
  }

  fail(req: object, err?: Error | ErrorWithCode) {
    const { host } = this.requestsStat.get(req) || {};

    statEvent({
      name: 'http.failure',
      payload: {
        host,
        value: this.getTiming(req),
        statusCode: (err as ErrorWithCode)?.code || 0,
      },
    });
  }

  getTiming(key: object) {
    const { start } = this.requestsStat.get(key) || {};

    return start ? getTimeDiff(start) : 0;
  }
}

const requestsStats = new RequestsStats();

function handleHttpRequest(request: ClientRequest, options: RequestOptions) {
  requestsStats.add(request, options);

  request.once('response', (response) => {
    response.once('error', (e) => {
      requestsStats.fail(request, e);
    });

    response.once('end', () => {
      const code = response.statusCode || 0;

      if (code >= 100 && code < 400) {
        requestsStats.success(request, response.statusCode || 0);
      } else {
        requestsStats.fail(request, { code });
      }
    });
  });
  request.once('error', (error) => {
    requestsStats.fail(request, error);
  });
}
