import { ExtensionConfig, MasterProcess, WorkerProcess } from 'luster';

import { WorkerStatMessage, getTimeDiff } from '@server/shared/stats';
import yasmkit from '@yandex-int/yasmkit';

import { getCPUUsage } from '../lib/cpuUsage';

const INTERVAL = 5000;

const toSnakeCase = (str: string) => str.replace(/https?:\/\//, '').replace(/[:\-.\/]+/g, '_');

// @see https://wiki.yandex-team.ru/golovan/tagsandsignalnaming/
const getSignalName = (name: string) => name.replace(/[^a-zA-Z0-9_\/\-./@]/g, '_');

type CountMetrics = Record<string, number>;
type TimeMetrics = Record<string, number[]>;

export interface StatServerConfig {
  port: number;
}

interface MasterStatMessage {
  wid: string;
  cpuUsage: number;
  memusage: NodeJS.MemoryUsage;
  latency: number;
  countMetrics: CountMetrics;
  timeMetrics: TimeMetrics;
}

module.exports = {
  configure: function (
    config: ExtensionConfig<StatServerConfig>,
    cluster: WorkerProcess | MasterProcess,
  ) {
    if (cluster.isMaster) {
      // eslint-disable-next-line @typescript-eslint/no-use-before-define
      return new Master(config, cluster);
    }

    if (cluster.isWorker) {
      // eslint-disable-next-line @typescript-eslint/no-use-before-define
      return new Worker(config, cluster);
    }
  },
};

class Master {
  private countMetricsSet = new Set<string>();
  private timeMetricsSet = new Set<string>();

  constructor(private config: ExtensionConfig<StatServerConfig>, private master: MasterProcess) {
    this.statServer();
    this.registerHandler();
  }

  statServer() {
    yasmkit.server.run({
      port: this.config.get('port'),
    });

    process.on('yasm', (data) => {
      if (data.name === 'custom.sum') {
        this.yasmSumEvent(`custom_${data.signalName}`, data.value);
      }

      if (data.name === 'custom.hist') {
        this.yasmHistEvent(`custom_${data.signalName}`, data.value);
      }
    });
  }

  /**
   * Собираем словарь всех возможных метрик
   * Он нужен, чтобы всегда отсылать все метрики даже с нулевым значением,
   * чтобы графики были непрерывны
   *
   * @param {CountMetrics} countMetrics
   * @param {TimeMetrics} timeMetrics
   * @memberof Master
   */
  gatherMetrics(countMetrics: CountMetrics, timeMetrics: TimeMetrics) {
    // eslint-disable-next-line array-callback-return
    Object.keys(countMetrics).forEach((item) => this.countMetricsSet.add(item));
    // eslint-disable-next-line array-callback-return
    Object.keys(timeMetrics).forEach((item) => this.timeMetricsSet.add(item));
  }

  /**
   * В пакет отправляемых метрик добавляем недостающие
   *    с нулевым значением или пустым набором
   *
   * @param {CountMetrics} countMetrics
   * @param {TimeMetrics} timeMetrics
   * @memberof Master
   */
  normalizeMetrics(countMetrics: CountMetrics, timeMetrics: TimeMetrics) {
    this.countMetricsSet.forEach((item) => {
      if (!countMetrics[item]) {
        countMetrics[item] = 0;
      }
    });

    this.timeMetricsSet.forEach((item) => {
      if (!timeMetrics[item]) {
        timeMetrics[item] = [];
      }
    });
  }

  yasmSumEvent(signal: string, value: number) {
    const signalName = getSignalName(signal);

    if (!yasmkit.metrics.has(signalName)) {
      yasmkit.metrics.addSummLine(signalName, { resetOnRead: false });
    }
    yasmkit.addEvent(signalName, value);
  }

  yasmHistEvent(signal: string, values: number[]) {
    const signalName = getSignalName(signal);

    if (!yasmkit.metrics.has(signalName)) {
      yasmkit.metrics.addHistogram(signalName);
    }
    values.forEach((value) => {
      yasmkit.addEvent(signalName, value);
    });
  }

  registerHandler() {
    const workers = this.master.getWorkersArray();

    this.master.registerRemoteCommand<MasterStatMessage>('stats', (sender, message) => {
      const { wid, cpuUsage, memusage, latency, countMetrics, timeMetrics } = message;

      this.yasmSumEvent(`cpu_usage_w${wid}`, cpuUsage);
      this.yasmSumEvent(`mem_usage_rss_w${wid}`, memusage.rss);
      this.yasmSumEvent(`mem_usage_heapTotal_w${wid}`, memusage.heapTotal);
      this.yasmSumEvent(`mem_usage_heapUsed_w${wid}`, memusage.heapUsed);
      this.yasmSumEvent(`latency_w${wid}`, latency);

      this.gatherMetrics(countMetrics, timeMetrics);
      this.normalizeMetrics(countMetrics, timeMetrics);

      Object.entries(countMetrics).forEach(([name, value]) => {
        this.yasmSumEvent(name, value);
      });

      Object.entries(timeMetrics).forEach(([name, value]) => {
        this.yasmHistEvent(name, value);
      });
    });

    setInterval(() => {
      workers.forEach((worker) => {
        // Не опрашиваем воркер, если в этот момент он рестартует.
        if (!worker.ready) {
          return;
        }

        worker.remoteCall('health');
      });
    }, INTERVAL).unref();
  }
}

class Worker {
  private countMetrics: CountMetrics;
  private timeMetrics: TimeMetrics;

  constructor(private config: ExtensionConfig<StatServerConfig>, private worker: WorkerProcess) {
    this.initMetrics();
    this.initStatusCollector();
  }

  addCountEvent(name: string, count = 1) {
    if (!this.countMetrics[name]) {
      this.countMetrics[name] = count;
    } else {
      this.countMetrics[name] += count;
    }
  }

  addTimeEvent(name: string, value: number) {
    if (!this.timeMetrics[name]) {
      this.timeMetrics[name] = [value];
    } else {
      this.timeMetrics[name].push(value);
    }
  }

  initMetrics() {
    this.countMetrics = {};
    this.timeMetrics = {};
  }

  initStatusCollector() {
    process.on('yasm', (data: WorkerStatMessage) => {
      const { name, payload } = data;
      const { statusCode, value, signalName } = payload || {};

      let { host, path } = payload || {};

      if (host) {
        host = toSnakeCase(host);
      }

      if (path) {
        path = toSnakeCase(path);
      }

      if (name === 'request.start') {
        this.addCountEvent('requests-count');

        if (path) {
          this.addCountEvent(`requests-count-${path}`);
        }
      }

      const code = statusCode?.toString() || '';

      if (name === 'request.finish' && code && path) {
        this.addCountEvent(`requests-finish-${path}-${code}`);

        if (value) {
          this.addTimeEvent(`requests-${path}-${code}`, value);
        }

        return;
      }

      if (name === 'http.success' && code && value && host) {
        return this.addTimeEvent(`success-${host}-${code}`, value);
      }

      if (name === 'http.failure' && host) {
        this.addCountEvent(`http-${host}-failure`);

        if (value) {
          this.addTimeEvent(`failure-http-${host}-${code}`, value);
        }

        if (code) {
          this.addCountEvent(`http-${host}-failure-${code}`);
        }

        return;
      }

      if (name === 'http.attempt' && value && host) {
        return this.addCountEvent(`http-${host}-attempt-${value}`);
      }

      if (name === 'http.retryRatio' && value && host) {
        return this.addCountEvent(`http-${host}_retry_ratio`, value);
      }

      if (name === 'custom.sum' && signalName && value) {
        return this.addCountEvent(signalName, value);
      }

      if (name === 'custom.hist' && signalName && value) {
        return this.addTimeEvent(signalName, value);
      }
    });

    this.worker.registerRemoteCommand('health', async (sender) => {
      const [cpuUsage, latency] = await Promise.all([getCPUUsage(), latencyCheck()]);

      // в ответ на запрос мастера шлём ему свои метрики
      sender.remoteCall<[MasterStatMessage]>('stats', {
        wid: this.worker.wid,
        latency,
        cpuUsage,
        memusage: process.memoryUsage(),
        countMetrics: this.countMetrics,
        timeMetrics: this.timeMetrics,
      });

      // каждый новый цикл начинаем с пустого набора метрик
      this.initMetrics();
    });
  }
}

function latencyCheck() {
  return new Promise<number>((resolve) => {
    const start = process.hrtime.bigint();

    setImmediate((start) => resolve(getTimeDiff(start)), start);
  });
}
