import path from 'path';

import luster from 'luster';

import { clusterConfig } from '@server/config/luster';

import { getPort } from '../config';
import { ClusterStatusConfig } from './extensions/cluster-status';

function startCluster() {
  const config = clusterConfig;

  const extensions: Record<string, object> = {
    'cluster-status': {
      processNamePrefix: config.processNamePrefix,
      readyThreshold: config.readyThreshold,
      workerPingInterval: config.workerPingInterval,
      workerTimeout: config.workerTimeout,
      ignoreException: true,
    } as ClusterStatusConfig,
    'http-stats': {},
    'stat-server': {
      port: config.yasmServerPort,
    },
    // https://a.yandex-team.ru/arc/trunk/arcadia/classifieds/nodejs/luster-bunker?rev=r9428796
    '@vertis/luster-bunker': {
      // интервал обновления в ms (по умолчанию, 2 минуты)
      refresh: 2 * 60000,
      // путь до кеша, созданного luster-bunker-precache
      // cachePath: 'path/to/cache.json',
      // опции для запроса http-бункера (отправляются в asker как есть)
      connection: {
        family: 6,
        timeout: config.bunkerTimeout,
        maxRetries: 10,
        requestId: 'luster-bunker',
        host: config.bunkerApi,
      },
      // массив нод, которые надо рекурсивно загрузить
      nodes: config.bunkerNodes,
    },
  };

  // Если LogFile отключен явно, то даже не загружаем этот extension
  // это, например, позволяет deploy-yandex-logger правильно формировать вывод для деплоя
  if (config.logFile !== false) {
    // https://a.yandex-team.ru/arc/trunk/arcadia/classifieds/nodejs/luster-log-file?rev=r9428796
    extensions['luster-log-file'] = {
      // override `console.log`, `.warn`, `.info` and `.error` methods
      // to add severity marks to output
      extendConsole: true,

      // logs files, both optional
      //   {string} fileName – stream output to file
      //   true – don't redirect output, keep as is
      //   false – shut down output
      stdout: config.logFile,
      stderr: config.logFile,
    };
  }

  // eslint-disable-next-line @yandex-id/i18n/no-unwrapped-strings
  luster.on('error', (err) => err && console.error('Ошибка инициализации мастера', err));

  process.on('SIGPIPE', () => {
    luster.emit('reopen-logs');
  });

  const app = luster.configure({
    app: __dirname + '/../main.js',
    workers: config.workers,
    control: {
      forkTimeout: config.forkTimeout,
      stopTimeout: config.stopTimeout,
      exitThreshold: config.exitThreshold,
      allowedSequentialDeaths: config.allowedSequentialDeaths,
    },
    server: {
      port: getPort(),
      groups: 1,
    },
    extensionsPath: path.join(__dirname, 'extensions'),
    extensions: extensions,
  });

  app.run();
}

startCluster();
