import merge from 'merge-options';

import { YEnv } from '@shared/types/env';

export type EnvConfig<T> = {
  common?: T;
  development?: Partial<T>;
  production?: Partial<T>;
  rc?: Partial<T>;
  testing?: Partial<T>;
  stress?: Partial<T>;
};

type EnvKeys = 'common' | YEnv;

export function getEnvConfig<T>(config: EnvConfig<T>): T {
  const env = getYEnv();
  // eslint-disable-next-line @typescript-eslint/no-use-before-define
  const stages = configMap[env];
  const partialConfigs = [];

  if (!stages) {
    throw new Error(`Unexpected config map for ${env} environment.`);
  }

  for (const stage of stages) {
    partialConfigs.push(config[stage]);
  }

  const resultConfig = merge(...partialConfigs);

  return resultConfig;
}

export function getYEnv(): YEnv {
  switch (process.env.YENV) {
    case 'local':
    case 'development':
      return 'development';
    case 'test':
    case 'testing':
      return 'testing';
    case 'stress':
      return 'stress';
    case 'rc':
      return 'rc';
    default:
      return 'production';
  }
}

const configMap: Record<YEnv, EnvKeys[]> = {
  development: ['common', 'development'],
  rc: ['common', 'production', 'rc'],
  production: ['common', 'production'],
  testing: ['common', 'testing'],
  stress: ['common', 'testing', 'stress'],
};

export function getPort(): number {
  const envPort = Number(process.env.port || process.env.PORT);

  return envPort > 0 ? envPort : 3000;
}

export const isDevMode = getYEnv() === 'development';

export const isWatchMode = process.env.NODE_ENV === 'development';
