import os from 'os';

import { LusterConfigControl } from 'luster';

import { ClusterStatusConfig } from '@server/cluster/extensions/cluster-status';
import { Nodes } from '@vertis/luster-bunker';

import { EnvConfig, getEnvConfig, getYEnv } from './env-config';
import { BunkerConfigService } from './services';

// eslint-disable-next-line dot-notation
export interface ClusterOptions extends LusterConfigControl, ClusterStatusConfig {
  yasmServerPort: number;
  workers?: number;
  logFile: boolean | string;
  bunkerApi: string;
  bunkerTimeout: number;
  bunkerNodes: Nodes;
}

function getNumberOrDefault(val: string | undefined, def: number): number {
  const parsedValue = Number(val);

  return parsedValue > 0 ? parsedValue : def;
}

function getBooleanOrDefault(val: string | undefined, def: string) {
  if (!val) {
    return def;
  }

  if (/^true$/i.test(val)) {
    return true;
  }

  if (/^false$/i.test(val)) {
    return false;
  }

  return val;
}

const yenv = getYEnv();
let workers;

if (yenv === 'production') {
  // Не разрешаем сжирать 100% ресурсов железа
  // Всего воркеров половина от процессоров, но не меньше 1 и не больше 8
  const cpusCount = Math.max(8, Math.min(1, Math.floor(os.cpus().length / 2)));

  workers = getNumberOrDefault(process.env.WORKERS, cpusCount);
} else {
  workers = getNumberOrDefault(process.env.WORKERS, 1);
}

// Ждем половину воркеров, но не менее одного
const readyThreshold = Math.max(Math.ceil(workers / 2), 1);

const { options: bunkerConfig } = new BunkerConfigService();

const config: EnvConfig<ClusterOptions> = {
  common: {
    processNamePrefix: 'id-front',
    yasmServerPort: 3011,
    workerPingInterval: getNumberOrDefault(process.env.WORKER_PING_INTERVAL, 10_000),
    workerTimeout: getNumberOrDefault(process.env.WORKER_TIMEOUT, 8_000),
    forkTimeout: getNumberOrDefault(process.env.FORK_TIMEOUT, 10_000),
    stopTimeout: 3_000,

    exitThreshold: 5_000,
    allowedSequentialDeaths: 3,
    workers: 1,
    readyThreshold: 1,
    ignoreException: true,
    logFile: true,
    bunkerApi: bunkerConfig.apiHost,
    bunkerTimeout: bunkerConfig.timeout,
    bunkerNodes: [
      {
        node: bunkerConfig.ohioProject,
        version: bunkerConfig.version,
      },
    ],
  },
  testing: {
    workers,
    readyThreshold,
    logFile: getBooleanOrDefault(process.env.LOGFILE, '/var/log/yandex/id-frontend/service.log'),
  },
  production: {
    workers,
    readyThreshold,
    logFile: getBooleanOrDefault(process.env.LOGFILE, '/var/log/yandex/id-frontend/service.log'),
  },
};

export const clusterConfig = getEnvConfig<ClusterOptions>(config);
