import { Injectable } from '@nestjs/common';
import { AUTO_TLD, CSPOptions, Directive, presets } from '@server/shared/csp';

import { EnvConfig, getEnvConfig } from '../env-config';

const config: EnvConfig<CSPOptions> = {
  common: {
    presets: {
      // base csp values
      NONE: presets.none([Directive.DEFAULT_SRC, Directive.BASE_URI]),
      SELF: presets.self([
        Directive.SCRIPT_SRC,
        Directive.STYLE_SRC,
        Directive.IMG_SRC,
        Directive.FONT_SRC,
        Directive.CONNECT_SRC,
        Directive.FRAME_ANCESTORS,
        Directive.FRAME_SRC,
        Directive.CHILD_SRC,
        Directive.PREFETCH_SRC,
      ]),
      DATA: presets.data([Directive.IMG_SRC, Directive.FONT_SRC, Directive.MEDIA_SRC]),
      BLOB: presets.blob([Directive.IMG_SRC, Directive.FRAME_SRC, Directive.CHILD_SRC]),
      UNSAFE_INLINE: presets.unsafeInline(),
      UNSAFE_EVAL: presets.usafeEval([Directive.SCRIPT_SRC, Directive.STYLE_SRC]),
      NONCE: presets.nonce(),

      // services
      YA_STATIC: presets.yaStatic(),
      AVATARS_MDS: presets.avatars(),
      MAIL: presets.mail(),
      METRIKA: presets.metrika(['ru']),
      EDADEAL: presets.edadeal(),
      GEO_SUGGEST: presets.geoSuggest(),
      STATIC_MAPS: presets.staticMaps(),
      DISK: presets.disk(),
      FORMS: presets.forms(),
      USER_ID: presets.userId(),
      S3: presets.s3(),
      MESSENGER_FRONT: presets.messengerFront('production'),
    },

    enableDefaultReportUri: true,
    defaultReportUriParams: {
      project: 'yandex-id',
    },
  },
  development: {
    presets: {
      AVATARS_MDS: presets.avatars('development'),
      GRAPHQL_PLAYGROUND: presets.graphQLPlayground(),
      GIBDD: presets.gibdd('development'),
      TRUST: presets.trust('development'),
      WSS: {
        [Directive.CONNECT_SRC]: [`wss://localhost.msup.yandex.${AUTO_TLD}`],
      },
    },
    defaultReportUriParams: {
      from: 'yandex-id-development',
    },
  },
  testing: {
    presets: {
      AVATARS_MDS: presets.avatars('testing'),
      GIBDD: presets.gibdd('testing'),
      TRUST: presets.trust('testing'),
    },
    reportOnly: true,
    defaultReportUriParams: {
      from: 'yandex-id-testing',
    },
  },
  production: {
    presets: {
      AVATARS_MDS: presets.avatars('production'),
      TRUST: presets.trust(),
      GIBDD: presets.gibdd(),
    },
    reportOnly: true,
    defaultReportUriParams: {
      from: 'yandex-id-production',
    },
  },
};

@Injectable()
export class CSPConfigService {
  createCSPOptions() {
    return getEnvConfig(config);
  }
}
