import { Injectable } from '@nestjs/common';
import { GeobaseService } from '@server/shared/geobase';
import { PassportService } from '@server/shared/passport';
import {
  AccountAuthentication,
  AccountIp,
  AccountEventAction as RawAccountAction,
  AccountAuthEvent as RawAccountAuthEvent,
  AccountAuthType as RawAccountAuthType,
  AccountEvent as RawAccountEvent,
  RestoreAction as RawRestoreAction,
} from '@server/shared/passport/api-types';
import { StaticMapService } from '@server/shared/static-map';
import {
  AccountAction,
  AccountActionType,
  AccountAuthType,
  AccountGeolocation,
  AccountLoginAction,
  AccountRestoreBy,
  AccountRestoreByType,
  BaseAccountAction,
} from '@shared/types/account';
import { RegionType } from '@yandex-int/yandex-geobase';

const AUTH_TYPES: Record<RawAccountAuthType, AccountAuthType> = {
  calendar: AccountAuthType.Calendar,
  imap: AccountAuthType.IMAP,
  pop3: AccountAuthType.POP3,
  smtp: AccountAuthType.SMTP,
  web: AccountAuthType.Web,
  webdav: AccountAuthType.WebDav,
  xmpp: AccountAuthType.XMPP,
  'password-oauth': AccountAuthType.PasswordOAuth,
};

const CHANGED_FIELD_ORDER = [
  'firstname',
  'lastname',
  'display_name',
  'birthday',
  'sex',
  'country',
  'city',
  'tz',
];

@Injectable()
export class AccountActionsService {
  constructor(
    private passportService: PassportService,
    private geobaseService: GeobaseService,
    private staticMapService: StaticMapService,
  ) {}

  async getActions(period: number, lang: string) {
    const [accountAuthEvents, accountEvents] = await Promise.all([
      this.passportService.getAccountAuthEvents(period),
      this.passportService.getAccountEvents(period),
    ]);

    const result: AccountAction[] = [];

    if (accountAuthEvents.status === 'ok') {
      const actions = Array.from(this.prepareAccountAuthActions(accountAuthEvents.events, lang));

      result.push(...actions);
    }

    if (accountEvents.status === 'ok') {
      const actions = Array.from(this.prepareAccountActions(accountEvents.events, lang));

      result.push(...actions);
    }

    return result.sort((a, b) => b.timestamp - a.timestamp);
  }

  private *prepareAccountActions(
    events: RawAccountEvent[],
    lang: string,
  ): Generator<AccountAction> {
    for (const event of events) {
      const { actions } = event;

      for (const action of actions) {
        if (
          event.event_type === 'app_passwords_disabled' &&
          action.type !== 'app_passwords_disabled'
        ) {
          continue;
        }

        yield this.prepareAccountAction(event, action, lang);
      }
    }
  }

  private *prepareAccountAuthActions(events: RawAccountAuthEvent[], lang: string) {
    for (const event of events) {
      const { authentications } = event;

      for (const authentication of authentications) {
        yield this.prepareAccountAuthAction(event, authentication, lang);
      }
    }
  }

  private prepareGeolocation(ipInfo: AccountIp, lang: string): AccountGeolocation {
    const { ip, geoid } = ipInfo;
    const geo: AccountGeolocation = {};

    if (ip) {
      geo.ip = ip;
    }

    let geoId = geoid;

    if (ip && !geoId) {
      geoId = this.geobaseService.getRegionIdByIp(ip);
    }

    if (geoId) {
      const region = this.geobaseService.getRegionById(geoId);
      const { nominative } = this.geobaseService.getLinguistics(geoId, lang);

      geo.region = nominative;

      if (region) {
        geo.coordinates = {
          lat: region.latitude,
          lng: region.longitude,
        };

        geo.mapUrl = this.staticMapService.getUrl({
          center: geo.coordinates,
          span: { lngDiff: 0.2, latDiff: 0.2 },
          size: { width: 600, height: 200 },
          lang,
        });
      }
    }

    return geo;
  }

  private prepareAccountAction(
    event: RawAccountEvent,
    action: RawAccountAction,
    lang: string,
  ): AccountAction {
    const { timestamp, browser, ip, os } = event;

    const baseAction: BaseAccountAction = {
      os: os.name && os.version ? { name: os.name, version: os.version } : undefined,
      browser:
        browser.name && browser.version
          ? { name: browser.name, version: browser.version }
          : undefined,
      geolocation: this.prepareGeolocation(ip, lang),
      timestamp: Math.ceil(timestamp * 1000),
    };

    switch (action.type) {
      case 'email_add':
        return {
          type: AccountActionType.EmailAdd,
          ...baseAction,
          emailBind: action.email_bind,
        };

      case 'email_remove':
        return {
          type: AccountActionType.EmailRemove,
          ...baseAction,
          emailUnbind: action.email_unbind,
        };

      case 'personal_data':
        const changedFields = action.changed_fields.map((key) => {
          return {
            key,
            value: this.prepareChangedField(key, action[key] as string | null, lang),
          };
        });

        const sortedChangedFields = changedFields.sort((a, b) => {
          return CHANGED_FIELD_ORDER.indexOf(a.key) - CHANGED_FIELD_ORDER.indexOf(b.key);
        });

        return {
          type: AccountActionType.PersonalData,
          ...baseAction,
          changedFields: sortedChangedFields,
        };

      case 'restore':
        return {
          type: AccountActionType.Restore,
          ...baseAction,
          restoreBy: this.prepareRestoreByAction(action),
        };

      case 'secure_phone_replace':
        return {
          type: AccountActionType.SecurePhoneReplace,
          ...baseAction,
          phoneSet: this.formatPhone(action.phone_set),
          phoneUnset: this.formatPhone(action.phone_unset),
        };

      case 'secure_phone_set':
        return {
          type: AccountActionType.SecurePhoneSet,
          ...baseAction,
          phoneSet: this.formatPhone(action.phone_set),
        };

      case 'secure_phone_unset':
        return {
          type: AccountActionType.SecurePhoneUnset,
          ...baseAction,
          phoneUnset: this.formatPhone(action.phone_unset),
        };

      case 'phone_bind':
        return {
          type: AccountActionType.PhoneBind,
          ...baseAction,
          phoneBind: this.formatPhone(action.phone_bind),
        };

      case 'phone_unbind':
        return {
          type: AccountActionType.PhoneUnbind,
          ...baseAction,
          phoneUnbind: this.formatPhone(action.phone_unbind),
        };

      case 'app_passwords_disabled':
        return { type: AccountActionType.AppPasswordsDisabled, ...baseAction };

      case 'app_passwords_enabled':
        return { type: AccountActionType.AppPasswordEnabled, ...baseAction };

      case 'app_passwords_revoked':
        return { type: AccountActionType.AppPasswordsRevoked, ...baseAction };

      case 'email_remove_all':
        return { type: AccountActionType.EmailRemoveAll, ...baseAction };

      case 'global_logout':
        return { type: AccountActionType.GlobalLogout, ...baseAction };

      case 'password_change':
        return { type: AccountActionType.PasswordChange, ...baseAction };

      case 'password_remove':
        return { type: AccountActionType.PasswordRemove, ...baseAction };

      case 'questions_change':
        return { type: AccountActionType.QuestionsChange, ...baseAction };

      case 'questions_remove':
        return { type: AccountActionType.QuestionsRemove, ...baseAction };

      case 'tokens_revoked':
        return { type: AccountActionType.TokensRevoked, ...baseAction };

      case 'totp_disabled':
        return { type: AccountActionType.TotpDisabled, ...baseAction };

      case 'totp_enabled':
        return { type: AccountActionType.TotpEnabled, ...baseAction };

      case 'totp_migrated':
        return { type: AccountActionType.TotpMigrated, ...baseAction };

      case 'web_sessions_revoked':
        return { type: AccountActionType.WebSessionsRevoked, ...baseAction };
    }
  }

  private getTimezoneName(value: string, lang: string) {
    const timezones = this.geobaseService.getKnownTimezones();
    const tz = timezones.find((tz) => tz.name === value);
    const name = tz?.desc_langs_with_offset[lang].split(',').slice(0, 3).join(', ');

    return name ?? value;
  }

  private getCountryName(value: string, lang: string) {
    const countries = this.geobaseService.getRegionsByType(RegionType.COUNTRY);
    const country = countries.find((c) => c.iso_name.toLowerCase() === value);

    if (country) {
      switch (lang) {
        case 'en':
          return country.en_name;

        case 'ru':
          return country.name;

        default:
          const localizeCountry = this.geobaseService.getLinguistics(country.id, lang);

          return localizeCountry.nominative ?? country.en_name;
      }
    }

    return value;
  }

  private prepareChangedField(key: string, value: string | null, lang: string) {
    if (!value) {
      return null;
    }

    switch (key) {
      case 'tz':
        return this.getTimezoneName(value, lang);

      case 'country':
        return this.getCountryName(value, lang);

      default:
        return value;
    }
  }

  private prepareRestoreByAction(action: RawRestoreAction): AccountRestoreBy {
    switch (action.restore_by) {
      case 'email':
        return {
          type: AccountRestoreByType.Email,
          email: action.email,
        };

      case 'phone':
        return {
          type: AccountRestoreByType.Phone,
          phone: this.formatPhone(action.phone),
        };

      case 'link':
        return { type: AccountRestoreByType.Link };

      case 'hint':
        return { type: AccountRestoreByType.Hint };

      case 'phone_and_2fa_factor':
        return { type: AccountRestoreByType.PhoneAnd2faFactor };

      case 'semi_auto':
        return { type: AccountRestoreByType.SemiAuto };
    }
  }

  private prepareAccountAuthAction(
    event: RawAccountAuthEvent,
    authentication: AccountAuthentication,
    lang: string,
  ): AccountLoginAction {
    const { auth } = event;
    const { browser = {}, ip, os = {}, authtype } = auth;
    const { timestamp } = authentication;

    const action: AccountLoginAction = {
      type: AccountActionType.Login,
      authType: AUTH_TYPES[authtype],
      os: os.name && os.version ? { name: os.name, version: os.version } : undefined,
      browser:
        browser.name && browser.version
          ? { name: browser.name, version: browser.version }
          : undefined,
      geolocation: this.prepareGeolocation(ip, lang),
      timestamp: Math.ceil(timestamp * 1000),
    };

    return action;
  }

  private formatPhone(phone: string) {
    return phone.replace(/\s/g, '\u00A0').replace(/\-/g, '\u00A0').replace(/\*/g, '•');
  }
}
