import { Args, Mutation, Query, Resolver } from '@nestjs/graphql';
import {
  ProfileUpdateInput,
  PublicIdSuggestInput,
  PublicIdUpdateInput,
  PublicIdValidateInput,
  SendCodeToBindPrimaryPhoneInput,
  SendCodeToBindPrimaryPhoneResult,
  SendCodeToConfirmPhoneInput,
  SendCodeToConfirmPhoneResult,
  SetQuestionInput,
  ToggleLoginWithPasswordAndSmsInput,
  VerifyCodeToBindPrimaryPhoneInput,
  VerifyCodeToBindPrimaryPhoneResult,
  VerifyCodeToConfirmPhoneInput,
  VerifyCodeToConfirmPhoneResult,
} from '@server/graphql-schema';

import { AccountsService } from './accounts.service';
import { PhoneService } from './phone.service';
import { ProfileService } from './profile.service';
import { PublicIdService } from './publicId.service';
import { QuestionService } from './question.service';

@Resolver()
export class AccountsResolver {
  constructor(
    private accountService: AccountsService,
    private question: QuestionService,
    private phoneService: PhoneService,
    private publicId: PublicIdService,
    private profileService: ProfileService,
  ) {}

  @Query('account')
  async getAccount() {
    const account = await this.accountService.getAccount(true);

    return account;
  }

  @Query('accountAvailableQuestions')
  async getAvailableQuestions() {
    return this.question.getAvailableQuestions();
  }

  @Mutation('accountEnableLoginWithQR')
  async enableLoginWithQR() {
    const account = await this.accountService.changeLoginWithQR(true);

    return { account };
  }

  @Mutation('accountDisableLoginWithQR')
  async disableLoginWithQR() {
    const account = await this.accountService.changeLoginWithQR(false);

    return { account };
  }

  @Mutation('accountSetQuestion')
  async setQuestion(@Args('input') input: SetQuestionInput) {
    return this.question.setQuestion(input);
  }

  @Mutation('accountEnableLoginWithPasswordAndSms')
  async enableLoginWithPasswordAndSms(@Args('input') input: ToggleLoginWithPasswordAndSmsInput) {
    const result = await this.accountService.toggleLoginWithPasswordAndSms(input, true);

    return result;
  }

  @Mutation('accountDisableLoginWithPasswordAndSms')
  async disableLoginWithPasswordAndSms(@Args('input') input: ToggleLoginWithPasswordAndSmsInput) {
    const result = await this.accountService.toggleLoginWithPasswordAndSms(input, false);

    return result;
  }

  @Mutation('sendCodeToConfirmPhone')
  async sendCodeToConfirmPhone(
    @Args('input') input: SendCodeToConfirmPhoneInput,
  ): Promise<SendCodeToConfirmPhoneResult> {
    const result = await this.phoneService.sendCodeToConfirmPhone(input);

    return result;
  }

  @Mutation('verifyCodeToConfirmPhone')
  async verifyCodeToConfirmPhone(
    @Args('input') input: VerifyCodeToConfirmPhoneInput,
  ): Promise<VerifyCodeToConfirmPhoneResult> {
    const result = await this.phoneService.verifyCodeToConfirmPhone(input);

    return result;
  }

  @Mutation('sendCodeToBindPrimaryPhone')
  async sendCodeToBindPrimaryPhone(
    @Args('input') input: SendCodeToBindPrimaryPhoneInput,
  ): Promise<SendCodeToBindPrimaryPhoneResult> {
    const result = await this.phoneService.sendCodeToBindPrimaryPhone(input);

    return result;
  }

  @Mutation('verifyCodeToBindPrimaryPhone')
  async verifyCodeToBindPrimaryPhone(
    @Args('input') input: VerifyCodeToBindPrimaryPhoneInput,
  ): Promise<VerifyCodeToBindPrimaryPhoneResult> {
    const result = await this.phoneService.verifyCodeToBindPrimaryPhone(input);

    return result;
  }

  @Mutation('validatePublicId')
  async validatePublicId(@Args('input') input: PublicIdValidateInput) {
    return await this.publicId.validatePublicId(input);
  }

  @Mutation('profileUpdate')
  async profileUpdate(@Args('input') input: ProfileUpdateInput) {
    return await this.profileService.updateProfile(input);
  }

  @Mutation('suggestPublicId')
  async suggestPublicId(@Args('input') input: PublicIdSuggestInput) {
    return await this.publicId.suggestPublicId(input);
  }

  @Mutation('updatePublicId')
  async updatePublicId(@Args('input') input: PublicIdUpdateInput) {
    return await this.publicId.updatePublicId(input);
  }
}
