import { Injectable } from '@nestjs/common';
import { Email } from '@server/graphql-schema';
import {
  Account,
  AccountType,
  AuthType,
  EmailType,
  PasswordInfo,
  Question,
  ToggleLoginWithPasswordAndSmsInput,
  ToggleLoginWithPasswordAndSmsPayload,
  ToggleLoginWithPasswordAndSmsProblem,
  ToggleLoginWithPasswordAndSmsProblemReason,
} from '@server/graphql-schema';
import { BlackboxService } from '@server/shared/blackbox';
import { BLACKBOX_ALIASES } from '@server/shared/blackbox/constants';
import { fromModel } from '@server/shared/libs';
import { LoggerService } from '@server/shared/logger';
import { PassportService } from '@server/shared/passport';
import { BlackboxResult } from '@yandex-int/nest-infra';

import {
  ConfirmationTrackStateException,
  ConfirmationTrackStateExceptionReason,
  ToggleLoginWithPasswordAndSmsException,
  ToggleLoginWithPasswordAndSmsReason,
} from './exceptions';
import { PhoneService } from './phone.service';

@Injectable()
export class AccountsService {
  constructor(
    private bb: BlackboxService,
    private passport: PassportService,
    private phoneService: PhoneService,
    private logger: LoggerService,
  ) {}

  async getAccount(shouldForceCache = false) {
    const blackbox = await this.bb.getBlackbox(shouldForceCache);
    const phones = await this.phoneService.getPhones(shouldForceCache);

    const accountType = this.getAccountType(blackbox);
    const isSuperLite = accountType === AccountType.SUPERLITE;
    const isNeoPhonish = accountType === AccountType.NEOPHONISH;
    const canLoginWithQR = !isNeoPhonish && !isSuperLite && !blackbox.isQrDisabled;
    const emails = this.getEmails(blackbox);
    const question = this.getQuestion(blackbox.question);

    return fromModel(Account, {
      id: blackbox.uid as string,
      type: accountType,
      authType: this.getAuthType(blackbox, accountType),
      canLoginWithQR,
      passwordInfo: this.getPasswordInfo(blackbox),
      emails,
      phones,
      question,
      publicId: blackbox.raw.public_id,
    });
  }

  async changeLoginWithQR(authWithQr: boolean) {
    await this.passport.updateAccount({ authWithQr });
    const account = await this.getAccount(true);

    return account;
  }

  async toggleLoginWithPasswordAndSms(input: ToggleLoginWithPasswordAndSmsInput, value: boolean) {
    try {
      await this.phoneService.validateConfirmationTrackState(input.trackId);
      await this.phoneService.toggleLoginWithPasswordAndSms(value);

      return fromModel(ToggleLoginWithPasswordAndSmsPayload, {
        account: await this.getAccount(true),
      });
    } catch (err) {
      this.logger.error(
        '[ToggleLoginWithPasswordAndSmsProblem]: an internal server error has occurred',
        { err },
      );

      if (err instanceof ConfirmationTrackStateException) {
        const reasons = {
          [ConfirmationTrackStateExceptionReason.PRIMARY_PHONE_NOT_BOUND]:
            ToggleLoginWithPasswordAndSmsProblemReason.PRIMARY_PHONE_NOT_BOUND,
          [ConfirmationTrackStateExceptionReason.PRIMARY_PHONE_NOT_CONFIRMED]:
            ToggleLoginWithPasswordAndSmsProblemReason.PRIMARY_PHONE_NOT_CONFIRMED,
          [ConfirmationTrackStateExceptionReason.TRACK_ID_INVALID]:
            ToggleLoginWithPasswordAndSmsProblemReason.TRACK_ID_INVALID,
          [ConfirmationTrackStateExceptionReason.TRACK_INVALID_STATE]:
            ToggleLoginWithPasswordAndSmsProblemReason.TRACK_INVALID_STATE,
        };

        return fromModel(ToggleLoginWithPasswordAndSmsProblem, {
          reason: reasons[err.reason],
        });
      }

      if (err instanceof ToggleLoginWithPasswordAndSmsException) {
        const reasons = {
          [ToggleLoginWithPasswordAndSmsReason.PRIMARY_PHONE_NOT_FOUND]:
            ToggleLoginWithPasswordAndSmsProblemReason.PRIMARY_PHONE_NOT_BOUND,
          [ToggleLoginWithPasswordAndSmsReason.UNKNOWN]:
            ToggleLoginWithPasswordAndSmsProblemReason.INTERNAL,
        };

        return fromModel(ToggleLoginWithPasswordAndSmsProblem, {
          reason: reasons[err.reason],
        });
      }

      return fromModel(ToggleLoginWithPasswordAndSmsProblem, {
        reason: ToggleLoginWithPasswordAndSmsProblemReason.INTERNAL,
      });
    }
  }

  private getQuestion(serializedQuestion = '') {
    const [id, text] = serializedQuestion.split(':');

    if (id && text) {
      return fromModel(Question, { id, text });
    }

    return null;
  }

  private getEmails(blackbox: BlackboxResult) {
    const emails = blackbox.addressList || [];

    return emails.map((email) => {
      const {
        address,
        native,
        validated: isConfirmed,
        default: isPrimary,
        unsafe: isUnsafe,
      } = email;
      const type = native ? EmailType.NATIVE : EmailType.EXTERNAL;

      return fromModel(Email, {
        type,
        address,
        isPrimary,
        isConfirmed,
        email: address,
        isUnsafe,
      });
    });
  }

  private getPasswordInfo(blackbox: BlackboxResult) {
    if (blackbox.havePassword && blackbox.lastPasswordUpdate !== '0') {
      return fromModel(PasswordInfo, {
        updatedAt: new Date(parseInt(blackbox.lastPasswordUpdate) * 1000),
      });
    }

    return null;
  }

  private getAuthType(blackbox: BlackboxResult, accountType: AccountType) {
    const { havePassword = false, is2faWithSms, is2faWithKey } = blackbox;
    const isSms2faEnabled = is2faWithSms === '1';
    const is2faEnabled = is2faWithKey === '1';

    if (accountType === AccountType.NEOPHONISH) {
      return AuthType.SMS_ONLY;
    }

    if (is2faEnabled) {
      return AuthType.YANDEX_KEY;
    }

    if (havePassword) {
      return isSms2faEnabled ? AuthType.PASSWORD_AND_SMS : AuthType.PASSWORD_ONLY;
    }

    return null;
  }

  private getAccountType(blackbox: BlackboxResult) {
    const { havePassword = false, raw } = blackbox;
    const { aliases } = raw;

    // NOTE: Порядок определения типа аккаунта имеет значение.
    if (aliases.hasOwnProperty(BLACKBOX_ALIASES.portal)) {
      return AccountType.NORMAL;
    }

    // должен стоять выше pdd
    if (aliases.hasOwnProperty(BLACKBOX_ALIASES.federal)) {
      return AccountType.FEDERAL;
    }

    if (aliases.hasOwnProperty(BLACKBOX_ALIASES.pdd)) {
      return AccountType.PDD;
    }

    if (aliases.hasOwnProperty(BLACKBOX_ALIASES.lite)) {
      if (!havePassword) {
        return AccountType.SUPERLITE;
      }

      return AccountType.LITE;
    }

    if (aliases.hasOwnProperty(BLACKBOX_ALIASES.mailish)) {
      return AccountType.MAILISH;
    }

    if (aliases.hasOwnProperty(BLACKBOX_ALIASES.social)) {
      return AccountType.SOCIAL;
    }

    if (aliases.hasOwnProperty(BLACKBOX_ALIASES.phonish)) {
      return AccountType.PHONISH;
    }

    if (aliases.hasOwnProperty(BLACKBOX_ALIASES.kinopoisk)) {
      return AccountType.KINOPOISK;
    }

    if (aliases.hasOwnProperty(BLACKBOX_ALIASES.uber)) {
      return AccountType.UBER;
    }

    if (aliases.hasOwnProperty(BLACKBOX_ALIASES.yambot)) {
      return AccountType.YAMBOT;
    }

    if (aliases.hasOwnProperty(BLACKBOX_ALIASES.kolonkish)) {
      return AccountType.KOLONKISH;
    }

    if (aliases.hasOwnProperty(BLACKBOX_ALIASES.neophonish)) {
      return AccountType.NEOPHONISH;
    }

    if (aliases.hasOwnProperty(BLACKBOX_ALIASES.scholar)) {
      return AccountType.SCHOLAR;
    }

    if (aliases.hasOwnProperty(BLACKBOX_ALIASES.kiddish)) {
      return AccountType.KIDDISH;
    }

    // NOTE: ситуация недостижимая в реальности
    throw new Error('Account without required aliases');
  }
}
