import { Injectable } from '@nestjs/common';
import {
  SendCodeToBindPrimaryPhoneInput,
  SendCodeToBindPrimaryPhonePayload,
  SendCodeToBindPrimaryPhoneProblem,
  SendCodeToBindPrimaryPhoneProblemReason,
  SendCodeToConfirmPhoneInput,
  SendCodeToConfirmPhonePayload,
  SendCodeToConfirmPhoneProblem,
  SendCodeToConfirmPhoneProblemReason,
  VerifyCodeToBindPrimaryPhoneInput,
  VerifyCodeToBindPrimaryPhonePayload,
  VerifyCodeToBindPrimaryPhoneProblem,
  VerifyCodeToBindPrimaryPhoneProblemReason,
  VerifyCodeToConfirmPhoneInput,
  VerifyCodeToConfirmPhonePayload,
  VerifyCodeToConfirmPhoneProblem,
  VerifyCodeToConfirmPhoneProblemReason,
} from '@server/graphql-schema';
import { Phone, PhoneOperation, PhoneOperationType } from '@server/graphql-schema';
import { BlackboxService } from '@server/shared/blackbox';
import { fromModel } from '@server/shared/libs';
import { PassportPhoneService } from '@server/shared/passport';

import {
  ConfirmationTrackStateException,
  ConfirmationTrackStateExceptionReason,
  ToggleLoginWithPasswordAndSmsException,
  ToggleLoginWithPasswordAndSmsReason,
} from './exceptions';

// TODO: По хорошему, нужно создать какую-то модель в модуле blackbox и унести такие вещи туда
const PHONE_OPERATION_TYPE_MAP: Record<string, PhoneOperationType> = {
  1: PhoneOperationType.BIND,
  2: PhoneOperationType.REMOVE,
  3: PhoneOperationType.SECURIFY,
  4: PhoneOperationType.REPLACE,
  5: PhoneOperationType.MARK,
  6: PhoneOperationType.ALIASIFY,
  7: PhoneOperationType.DEALIASIFY,
};

const PHONE_OPERATION_FLAGS = {
  ALIASIFY: 0,
  SHOULD_IGNORE_BINDING_LIMIT: 1,
  IN_QUARANTINE: 2,
};

@Injectable()
export class PhoneService {
  constructor(private bb: BlackboxService, private phoneService: PassportPhoneService) {}

  async getPhones(shouldForceCache = false) {
    const blackbox = await this.bb.getBlackbox(shouldForceCache);
    const operations = this.getPhoneOperations(blackbox.raw.phone_operations);
    const phones = (blackbox.phones ?? []).filter((phone) => phone.isBound === '1');

    return phones.map((phone) => {
      const operation = operations[phone.id] || null;

      return fromModel(Phone, {
        id: phone.id,
        number: phone.phone.replace(/‒/g, '-'),
        isPrimary: phone.isSecured === '1',
        isDefault: phone.isDefault === '1',
        operation,
      });
    });
  }

  async toggleLoginWithPasswordAndSms(value: boolean) {
    const response = await this.phoneService.toggleLoginWithPasswordAndSms(value);

    if (response.status === 'ok') {
      return;
    }

    for (const error of response.errors) {
      if (error === 'phone_secure.not_found') {
        throw new ToggleLoginWithPasswordAndSmsException(
          'The primary phone is not found',
          ToggleLoginWithPasswordAndSmsReason.PRIMARY_PHONE_NOT_FOUND,
        );
      }
    }

    throw new ToggleLoginWithPasswordAndSmsException(
      `Unknown error: ${JSON.stringify(response)}`,
      ToggleLoginWithPasswordAndSmsReason.UNKNOWN,
    );
  }

  async sendCodeToConfirmPhone(input: SendCodeToConfirmPhoneInput) {
    const result = await this.phoneService.sendCodeToConfirmPhone({
      number: input.number,
      code_format: 'by_3_dash',
      track_id: input.trackId || undefined,
    });

    if (result.status === 'ok') {
      return fromModel(SendCodeToConfirmPhonePayload, {
        trackId: result.track_id,
        // NOTE: Кажется, что правильней возвращать e164 и форматировать на клиенте
        // но сейчас нет никакого форматтера
        number: result.number.international,
        expiryTimestamp: new Date(result.deny_resend_until * 1000),
      });
    }

    const reason = this.resolveFirstProblemReason(
      result.errors,
      {
        'number.invalid': SendCodeToConfirmPhoneProblemReason.PHONE_NUMBER_INVALID,
        'phone.blocked': SendCodeToConfirmPhoneProblemReason.PHONE_NUMBER_BLOCKED,
        'sms_limit.exceeded': SendCodeToConfirmPhoneProblemReason.SMS_LIMIT_EXCEEDED,
      },
      SendCodeToConfirmPhoneProblemReason.INTERNAL,
    );

    return fromModel(SendCodeToConfirmPhoneProblem, { reason });
  }

  async verifyCodeToConfirmPhone(input: VerifyCodeToConfirmPhoneInput) {
    const result = await this.phoneService.verifyCodeToConfirmPhone({
      track_id: input.trackId,
      code: input.code,
    });

    if (result.status === 'ok') {
      return fromModel(VerifyCodeToConfirmPhonePayload, { trackId: input.trackId });
    }

    const reason = this.resolveFirstProblemReason(
      result.errors,
      {
        'track_id.invalid': VerifyCodeToConfirmPhoneProblemReason.TRACK_ID_INVALID,
        'code.invalid': VerifyCodeToConfirmPhoneProblemReason.CODE_INVALID,
        'sms.not_sent': VerifyCodeToConfirmPhoneProblemReason.CODE_NOT_SENT,
        'confirmations_limit.exceeded':
          VerifyCodeToConfirmPhoneProblemReason.CONFIRMATIONS_LIMIT_EXCEEDED,
      },
      VerifyCodeToConfirmPhoneProblemReason.INTERNAL,
    );

    return fromModel(VerifyCodeToConfirmPhoneProblem, { reason });
  }

  async sendCodeToBindPrimaryPhone(input: SendCodeToBindPrimaryPhoneInput) {
    const result = await this.phoneService.sendCodeToBindPrimaryPhone({
      track_id: input.trackId || undefined,
      number: input.number,
      code_format: 'by_3_dash',
    });

    if (result.status === 'ok') {
      return fromModel(SendCodeToBindPrimaryPhonePayload, {
        trackId: result.track_id,
        // NOTE: Кажется, что правильней возвращать e164 и форматировать на клиенте
        // но сейчас нет никакого форматтера
        number: result.number.international,
        expiryTimestamp: new Date(result.deny_resend_until * 1000),
      });
    }

    const reason = this.resolveFirstProblemReason(
      result.errors,
      {
        'number.invalid': SendCodeToBindPrimaryPhoneProblemReason.PHONE_NUMBER_INVALID,
        'phone.blocked': SendCodeToBindPrimaryPhoneProblemReason.PHONE_NUMBER_BLOCKED,
        'sms_limit.exceeded': SendCodeToBindPrimaryPhoneProblemReason.SMS_LIMIT_EXCEEDED,
      },
      SendCodeToBindPrimaryPhoneProblemReason.INTERNAL,
    );

    return fromModel(SendCodeToBindPrimaryPhoneProblem, { reason });
  }

  async verifyCodeToBindPrimaryPhone(input: VerifyCodeToBindPrimaryPhoneInput) {
    const result = await this.phoneService.verifyCodeToBindPrimaryPhone({
      track_id: input.trackId,
      code: input.code,
    });

    if (result.status === 'ok') {
      return fromModel(VerifyCodeToBindPrimaryPhonePayload, {
        trackId: input.trackId,
      });
    }

    const reason = this.resolveFirstProblemReason(
      result.errors,
      {
        'track_id.invalid': VerifyCodeToBindPrimaryPhoneProblemReason.TRACK_ID_INVALID,
        'code.invalid': VerifyCodeToBindPrimaryPhoneProblemReason.CODE_INVALID,
        'code.empty': VerifyCodeToBindPrimaryPhoneProblemReason.CODE_EMPTY,
        'sms.not_sent': VerifyCodeToBindPrimaryPhoneProblemReason.CODE_NOT_SENT,
        'confirmations_limit.exceeded':
          VerifyCodeToBindPrimaryPhoneProblemReason.CONFIRMATIONS_LIMIT_EXCEEDED,
      },
      VerifyCodeToBindPrimaryPhoneProblemReason.INTERNAL,
    );

    return fromModel(VerifyCodeToBindPrimaryPhoneProblem, { reason });
  }

  async validateConfirmationTrackState(trackId: string) {
    const { phones = [] } = await this.bb.getBlackbox();
    const primaryPhone = phones.find((phone) => phone.isBound === '1' && phone.isSecured === '1');

    if (!primaryPhone) {
      throw new ConfirmationTrackStateException(
        'The primary phone is not bound',
        ConfirmationTrackStateExceptionReason.PRIMARY_PHONE_NOT_BOUND,
      );
    }

    const trackState = await this.phoneService.getTrackState(trackId);

    if (trackState.status === 'error') {
      throw new ConfirmationTrackStateException(
        `Invalid track: ${JSON.stringify(trackState)}`,
        ConfirmationTrackStateExceptionReason.TRACK_ID_INVALID,
      );
    }

    if (
      (trackState.state !== 'confirm' && trackState.state !== 'confirm_and_bind_secure') ||
      primaryPhone.e164number !== trackState.phone_confirmation_phone_number
    ) {
      throw new ConfirmationTrackStateException(
        'Invalid track state',
        ConfirmationTrackStateExceptionReason.TRACK_INVALID_STATE,
      );
    }

    if (trackState.phone_confirmation_is_confirmed !== '1') {
      throw new ConfirmationTrackStateException(
        'The primary phone is not confirmed',
        ConfirmationTrackStateExceptionReason.PRIMARY_PHONE_NOT_CONFIRMED,
      );
    }
  }

  private resolveFirstProblemReason<T, K extends string = string>(
    errors: K[],
    reasons: Record<K, T>,
    defaultValue: T,
  ) {
    for (const error of errors) {
      const value = reasons[error];

      if (value) {
        return value;
      }
    }

    return defaultValue;
  }

  private getPhoneOperations(operationsMap?: Record<string, string | undefined>) {
    const operations: Record<string, PhoneOperation> = {};

    if (!operationsMap) {
      return operations;
    }

    for (const operationStr of Object.values(operationsMap)) {
      if (!operationStr) {
        continue;
      }

      // NOTE: Порядок значений в массиве имеет значение
      const [
        id,
        _uid,
        phoneId,
        _securityIdentity,
        type,
        started,
        finished,
        codeValue,
        _codeChecksCount,
        _codeSendCount,
        _codeLastSent,
        _codeConfirmed,
        _passwordVerified,
        rawFlags,
        _phoneId2,
      ] = operationStr.split(',');

      const flags = parseInt(rawFlags);
      const operation = fromModel(PhoneOperation, {
        id,
        type: PHONE_OPERATION_TYPE_MAP[type],
        startedAt: new Date(Number(started) * 1000),
        finishedAt: new Date(Number(finished) * 1000),
        isOwner: Boolean(codeValue),
        inQuarantine: ((flags >> PHONE_OPERATION_FLAGS.IN_QUARANTINE) & 1) === 1,
      });

      operations[phoneId] = operation;
    }

    return operations;
  }
}
