import { Injectable } from '@nestjs/common';
import {
  PublicIdSuggestInput,
  PublicIdSuggestPayload,
  PublicIdSuggestProblem,
  PublicIdUpdateInput,
  PublicIdUpdatePayload,
  PublicIdUpdateProblem,
  PublicIdValidateInput,
  PublicIdValidatePayload,
  PublicIdValidateProblem,
} from '@server/graphql-schema';
import { fromModel } from '@server/shared/libs';
import { LoggerService } from '@server/shared/logger/logger.service';
import { PassportPublicIdService, PassportService } from '@server/shared/passport';

@Injectable()
export class PublicIdService {
  private errorPrefix = 'public_id';

  constructor(
    private passport: PassportService,
    private passportPublicId: PassportPublicIdService,
    private logger: LoggerService,
  ) {}

  async validatePublicId(input: PublicIdValidateInput) {
    if (!input.trackId) {
      input.trackId = await this.passport.getTrackId();
    }

    const response = await this.passportPublicId.validatePublicId({
      publicId: input.publicId,
      trackId: input.trackId,
    });

    if (response.status === 'ok') {
      return fromModel(PublicIdValidatePayload, {
        trackId: response.id,
      });
    }

    return fromModel(PublicIdValidateProblem, {
      reason: this.errorHandler(response.errors),
    });
  }

  async updatePublicId(input: PublicIdUpdateInput) {
    if (!input.trackId) {
      input.trackId = await this.passport.getTrackId();
    }

    const response = await this.passport.updateProfile({
      publicId: input.publicId,
      trackId: input.trackId,
    });

    if (response.status === 'error') {
      return fromModel(PublicIdUpdateProblem, {
        reason: this.errorHandler(response.errors),
      });
    }

    return fromModel(PublicIdUpdatePayload, {});
  }

  async suggestPublicId(input: PublicIdSuggestInput) {
    if (!input.trackId) {
      input.trackId = await this.passport.getTrackId();
    }

    const response = await this.passport.suggestLogin({
      trackId: input.trackId,
      firstName: input.firstName ?? undefined,
      lastName: input.lastName ?? undefined,
      login: input.login ?? undefined,
      language: input.login ?? undefined,
    });

    if (response.status === 'error') {
      return fromModel(PublicIdSuggestProblem, {
        reason: this.errorHandler(response.errors),
      });
    }

    return fromModel(PublicIdSuggestPayload, {
      trackId: response.track_id,
      logins: response.suggested_logins,
    });
  }

  private errorHandler(errors: string[]) {
    const _errors = Array.isArray(errors) ? errors : [errors];

    const internalErrors: string[] = _errors.filter((error) => !error.startsWith(this.errorPrefix));

    if (internalErrors.length) {
      this.logger.error(internalErrors);

      return 'internal';
    }

    return _errors[0] || 'internal';
  }
}
