import qs from 'querystring';

import { Injectable, InternalServerErrorException } from '@nestjs/common';
import {
  AvailableQuestion,
  SetQuestionInput,
  SetQuestionPayload,
  SetQuestionProblem,
  SetQuestionProblemKind,
} from '@server/graphql-schema';
import { fromModel } from '@server/shared/libs';
import { LoggerService } from '@server/shared/logger';
import { PassportApiResponse, PassportService } from '@server/shared/passport';
import { Context } from '@yandex-int/nest-common';

import { QuestionsData } from './api-types';

@Injectable()
export class QuestionService {
  constructor(
    private passport: PassportService,
    private logger: LoggerService,
    private context: Context,
  ) {}

  async getAvailableQuestions(): Promise<AvailableQuestion[]> {
    const lang = this.context.req.langdetect.id;
    const response = await this.passport.http.post<PassportApiResponse<QuestionsData>>(
      `/1/questions/?consumer=passport`,
      qs.stringify({ display_language: lang }),
    );

    if (response.data.status === 'error') {
      throw new InternalServerErrorException(response.data.errors);
    }

    return response.data.questions.map((question) =>
      fromModel(AvailableQuestion, {
        id: question.id,
        text: question.value,
      }),
    );
  }

  async setQuestion(input: SetQuestionInput): Promise<SetQuestionPayload | SetQuestionProblem> {
    const lang = this.context.req.langdetect.id;

    if (!input.trackId) {
      input.trackId = await this.passport.getTrackId();
    }

    const payload: Record<string, string> = {
      track_id: input.trackId,
      new_answer: input.newAnswer,
      display_language: lang,
    };

    if (input.oldAnswer) {
      payload.answer = input.oldAnswer;
    }

    if (input.newQuestion) {
      payload.question = input.newQuestion;
    } else {
      payload.question_id = input.questionId;
    }

    const response = await this.passport.http.post<PassportApiResponse>(
      `/1/account/questions/change/?consumer=passport`,
      qs.stringify(payload),
      {
        headers: {
          'Ya-Client-Cookie': this.context.req.headers.cookie,
          'Ya-Client-Host': this.context.req.headers.host,
          'Ya-Client-User-Agent': this.context.req.headers['user-agent'],
          'Ya-Consumer-Client-Ip': this.context.req.ip,
        },
      },
    );

    if (response.data.status === 'error') {
      const { errors } = response.data;
      let reason = SetQuestionProblemKind.INTERNAL;

      if (errors.includes('password.required')) {
        reason = SetQuestionProblemKind.PASSWORD_REQUIRED;
      } else if (errors.includes('compare.not_matched')) {
        reason = SetQuestionProblemKind.COMPARE_NOT_MATCHED;
      } else {
        this.logger.error('SetQuestionError', { errors, err: errors[0] });
      }

      return fromModel(SetQuestionProblem, { trackId: input.trackId, reason });
    }

    return fromModel(SetQuestionPayload, {});
  }
}
