import { ApolloError } from 'apollo-server-express';

import { Injectable } from '@nestjs/common';
import {
  Address,
  AddressType,
  CreateOrUpdateAddressInput,
  CreateOrUpdateAddressPayload,
  CreateOrUpdateAddressProblem,
  CreateOrUpdateAddressProblemKind,
  DeleteAddressInput,
  DeleteAddressPayload,
  DeleteAddressProblem,
  DeleteAddressProblemKind,
} from '@server/graphql-schema';
import { BlackboxService } from '@server/shared/blackbox';
import { HttpService } from '@server/shared/http';
import { fromModel } from '@server/shared/libs';
import { LoggerService } from '@server/shared/logger';
import { StaticMapService } from '@server/shared/static-map';

import { IMAGE_HEIGHT, IMAGE_SCALE, IMAGE_WIDTH } from './address.constants';
import { AddressListResponse } from './address.interface';

@Injectable()
export class AddressService {
  constructor(
    private blackboxService: BlackboxService,
    private http: HttpService,
    private staticMapService: StaticMapService,
    private logger: LoggerService,
  ) {}

  async getAddressList() {
    try {
      const blackbox = await this.blackboxService.getBlackbox();

      const { data } = await this.http.get<AddressListResponse>('/address/list', {
        params: {
          user_id: blackbox.uid,
        },
      });

      // TODO: Возможно стоит добавить фильтрацию по owner_service.
      const addressList = data.addresses.map((address) => {
        const previewUrl = this.staticMapService.getUrl({
          center: {
            lat: address.location?.latitude,
            lng: address.location?.longitude,
          },
          size: {
            width: IMAGE_WIDTH,
            height: IMAGE_HEIGHT,
          },
          span: {
            latDiff: IMAGE_SCALE,
            lngDiff: IMAGE_SCALE,
          },
          hideText: true,
        });

        const region = [address.country, address.city].filter(Boolean).join(', ');
        const shortText = [address.street, address.building, address.room]
          .filter(Boolean)
          .join(', ');

        const addressModel = fromModel(Address, {
          id: address.id,
          comment: address.comment,
          entrance: address.entrance,
          floor: address.floor,
          fullText: address.address_line,
          intercom: address.intercom,
          previewUrl, // TODO: вынести в отдельный резолвер, т.к. требуется параметризация.
          region,
          room: address.room,
          shortText,
          street: address.street,
          type: this.getAddressType(address.type),
        });

        return addressModel;
      });

      return addressList;
    } catch (error) {
      this.logger.error('[AddressService] getAddressesList error', { err: error });

      throw new ApolloError('Failed to get a list of addresses', 'ADDRESSES_ERROR');
    }
  }

  async createOrUpdateAddress(input: CreateOrUpdateAddressInput) {
    // TODO: Возможно стоит добавить валидацию на заполненность fullText.
    // TODO: Нужно дополнить логику — можно создать только один домашний и рабочий адрес.
    try {
      const blackbox = await this.blackboxService.getBlackbox();

      const addressType = {
        [AddressType.Home]: 'home',
        [AddressType.Work]: 'work',
        [AddressType.Other]: 'other',
      }[input.type];

      const payload = {
        address_line: input.fullText,
        address_type: addressType,
        comment: input.comment,
        entrance: input.entrance,
        floor: input.floor,
        intercom: input.intercom,
        room: input.room,
      };

      const endpoint = input.id
        ? `/address/update?user_id=${blackbox.uid}&id=${input.id}&user_type=uid`
        : `/address/create?user_id=${blackbox.uid}&user_type=uid`;

      await this.http.post(endpoint, payload);

      return fromModel(CreateOrUpdateAddressPayload, {});
    } catch (error) {
      this.logger.error('[AddressService] createOrUpdateAddress error', { err: error });

      return fromModel(CreateOrUpdateAddressProblem, {
        reason: CreateOrUpdateAddressProblemKind.INTERNAL,
      });
    }
  }

  async deleteAddress(input: DeleteAddressInput) {
    try {
      const blackbox = await this.blackboxService.getBlackbox();

      await this.http.get(`/address/delete`, {
        params: {
          user_id: blackbox.uid,
          id: input.id,
          user_type: 'uid',
        },
      });

      return fromModel(DeleteAddressPayload, {});
    } catch (error) {
      this.logger.error('[AddressService] deleteAddress error', { err: error });

      return fromModel(DeleteAddressProblem, { reason: DeleteAddressProblemKind.INTERNAL });
    }
  }

  private getAddressType(type: string | null) {
    switch (type) {
      case 'work':
        return AddressType.Work;
      case 'home':
        return AddressType.Home;
      default:
        return AddressType.Other;
    }
  }
}
