import type { NextFunction, Request, Response } from 'express';
import { isString } from 'lodash';

import { Controller, Get, Next, Req, Res } from '@nestjs/common';

const complete3dsHtml = `<!doctype html>
    <html>
        <head>
            <title>YandexID.Payment::3DS-Complete</title>
            <meta charset="utf-8"/>
        </head>
        <body>
        <script>
            if (parent !== window && parent.postMessage) {
                parent.postMessage('{"source":"YandexIdPayment","type":"3DS_COMPLETE"}', "*");
            }
        </script>
        </body>
    </html>
`;

interface FormParams {
  action: string;
  method: 'get' | 'post';
}

const getFrame3dsHtml = (formParams: FormParams) => `<!doctype html>
  <html style="height: 100%;">
      <head>
          <title>YandexID.Payment::3DS</title>
          <meta charset="utf-8"/>
      </head>
      <body style="height: 100%;">
          <style>
              html,body,iframe {
                  display: block;
                  width: 100%;
                  min-width: 100%;
                  height: 100%;
                  min-height: 100%;
                  margin: 0;
                  padding: 0;
                  border: none;
              }
              form {
                  position: absolute;
                  opacity: 0;
              }
          </style>
          <form target="frame-3ds-inner" action="${formParams.action}" method="${formParams.method}">
              <input type="submit" value="send" />
          </form>
          <iframe name="frame-3ds-inner" id="frame-3ds-inner" title="3DS"></iframe>
          <script>
              window.addEventListener('message', function(ev) {
                  if (parent !== window && parent.postMessage) {
                      parent.postMessage(ev.data, "*");
                  }
              });

              var form = document.getElementsByTagName('form')[0];

              form.submit();
              form.style.display = 'none';
          </script>
      </body>
  </html>
`;

const getQuerySrc = (req: Request): string => {
  const { src } = req.query;

  if (!isString(src)) {
    throw new Error(`Invalid 3ds param - src="${JSON.stringify(src)}"`);
  }

  return src;
};

const getQueryMethod = (req: Request): 'post' | 'get' => {
  const method = req.query.method as string;

  if (!['post', 'get'].includes(method)) {
    throw new Error(`Invalid 3ds param - method="${JSON.stringify(method)}"`);
  }

  return method as 'post' | 'get';
};

@Controller()
export class BillsController {
  @Get('pay/payment/3ds-complete')
  complete3ds(@Res() res: Response) {
    return res.send(complete3dsHtml);
  }

  @Get('pay/payment/frame-3ds')
  frame3ds(@Req() req: Request, @Res() res: Response, @Next() next: NextFunction) {
    let html;

    try {
      const src = getQuerySrc(req);
      const method = getQueryMethod(req);

      html = getFrame3dsHtml({ action: src, method });
    } catch (err) {
      return next(err);
    }

    return res.send(html);
  }
}
