import { Injectable, InternalServerErrorException } from '@nestjs/common';
import {
  AddBillsDocumentInput,
  CreateOrderBillInput,
  DeleteBillsDocumentInput,
  StartOrderBillInput,
  UpdateBillsDocumentInput,
  getTransactionInput,
} from '@server/graphql-schema';
import { HttpService } from '@server/shared/http';
import { Context, RequestIdService } from '@yandex-int/nest-common';

import { billReducer, documentReducer } from './bills.helpers';
import {
  BillPayment,
  BillPaymentStatus,
  BillsData,
  BillsResponse,
  DeleteResponse,
  Document,
  DocumentData,
  DocumentsData,
  OrderData,
} from './bills.interface';

@Injectable()
export class BillsService {
  constructor(
    private context: Context,
    private http: HttpService,
    private requestIdService: RequestIdService,
  ) {}

  getDefaultOptions = () => {
    return {
      headers: {
        'X-Request-ID': this.requestIdService.getRequestId(),
      },
    };
  };

  async getBills(): Promise<BillsData> {
    try {
      const response = await this.http.get<BillsResponse<BillsData>>(
        '/api/v1/bills',
        this.getDefaultOptions(),
      );

      const { data } = response.data;

      return { state: data.state, bills: data.bills.map(billReducer) };
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  async restartSearch(): Promise<BillsResponse<{}>> {
    try {
      await this.http.post<BillsResponse<{}>>('/api/v1/search/bills', this.getDefaultOptions());

      return { status: 'ok', data: {} };
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  async getDocuments(): Promise<Document[]> {
    try {
      const response = await this.http.get<BillsResponse<DocumentsData>>(
        '/api/v1/documents',
        this.getDefaultOptions(),
      );

      return response.data.data.documents.map(documentReducer);
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  async addDocument({ type, value, title }: AddBillsDocumentInput): Promise<Document> {
    try {
      // Храним номер документа без пробелов
      const normalizedValue = value.replace(/ /g, '');

      const response = await this.http.post<BillsResponse<DocumentData>>(
        '/api/v1/documents',
        { type, value: normalizedValue, title },
        this.getDefaultOptions(),
      );

      return documentReducer(response.data.data.document);
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  async updateDocument({ type, value, title, id }: UpdateBillsDocumentInput): Promise<Document> {
    try {
      // Храним номер документа без пробелов
      const normalizedValue = value.replace(/ /g, '');

      const response = await this.http.put<BillsResponse<DocumentData>>(
        `/api/v1/documents/${id}`,
        { type, value: normalizedValue, title },
        this.getDefaultOptions(),
      );

      return documentReducer(response.data.data.document);
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  async deleteDocument({ id }: DeleteBillsDocumentInput): Promise<DeleteResponse> {
    try {
      const response = await this.http.delete<BillsResponse<{}>>(
        `/api/v1/documents/${id}`,
        this.getDefaultOptions(),
      );

      return { status: response.data.status };
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  async createOrder({ ids }: CreateOrderBillInput): Promise<any> {
    try {
      const response = await this.http.post<BillsResponse<OrderData>>(
        '/api/v1/orders',
        { bill_ids: ids },
        this.getDefaultOptions(),
      );

      return response.data.data.order;
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  async startOrder({
    id,
    payer_full_name,
    payment_token,
    return_url,
    payment_method,
    mpi_3ds_info,
  }: StartOrderBillInput): Promise<any> {
    try {
      const headers = this.context.headers;
      const ip = this.context.req.ip ?? headers['x-forwarded-for'];
      const browser_ip = Array.isArray(ip) ? ip[0] : ip.split(',')[0];
      const browser_accept_header = String(headers.accept);
      const browser_user_agent = String(headers['user-agent']);
      const browser_language = String(headers['accept-language']).split(',')[0];

      const response = await this.http.post<BillsResponse<BillPayment>>(
        `/api/v1/orders/${id}/transactions`,
        {
          payer_full_name,
          payment_token,
          return_url,
          payment_method,
          mpi_3ds_info: {
            ...mpi_3ds_info,
            browser_ip,
            browser_accept_header,
            browser_user_agent,
            browser_language,
            browser_javascript_enabled: true,
          },
        },
        this.getDefaultOptions(),
      );

      return response.data.data;
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  async getTransaction({ id }: getTransactionInput): Promise<DeleteResponse> {
    try {
      const response = await this.http.get<BillsResponse<BillPaymentStatus>>(
        `/api/v1/transactions/${id}`,
        this.getDefaultOptions(),
      );

      return response.data.data.transaction;
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }
}
