import { Injectable, InternalServerErrorException } from '@nestjs/common';
import { BlackboxService } from '@server/shared/blackbox';
import { BLACKBOX_PHONE_ATTRIBUTES } from '@server/shared/blackbox/blackbox.attributes';
import { HttpService } from '@server/shared/http';

import {
  BlackboxFamilyInfo,
  BlackboxUserInfoData,
  BlackboxUserPhonesData,
  UserInfo,
} from './blackbox-api.interface';

@Injectable()
export class BlackboxApiService {
  constructor(private http: HttpService, private blackboxService: BlackboxService) {}

  async getFamilyData() {
    const familyUids = await this.getFamilyUsers();

    if (!familyUids) {
      return undefined;
    }

    return await this.getUsersInfo(familyUids);
  }

  async getFamilyUsers(): Promise<string[] | undefined> {
    try {
      const blackbox = await this.blackboxService.getBlackbox();
      const familyId = blackbox.raw.family_info?.family_id;

      if (!familyId) {
        return undefined;
      }

      const response = await this.http.get<BlackboxFamilyInfo>('/blackbox', {
        params: {
          method: 'family_info',
          family_id: familyId,
          format: 'json',
          userip: '127.0.0.1',
        },
      });

      return response.data.family[familyId].users.map(({ uid }) => uid);
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  async checkSecurePhones(uids: string[]) {
    try {
      const response = await this.http.get<BlackboxUserPhonesData>('/blackbox', {
        params: {
          method: 'userinfo',
          uid: uids.join(','),
          userip: '127.0.0.1',
          format: 'json',
          getphones: 'all',
          phone_attributes: BLACKBOX_PHONE_ATTRIBUTES['phone.is_secured'],
        },
      });

      return response.data.users.reduce((acc, user) => {
        acc[user.id] = user.phones.some(
          (phone) => phone.attributes[BLACKBOX_PHONE_ATTRIBUTES['phone.is_secured']] === '1',
        );

        return acc;
      }, {} as Record<string, boolean>);
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  async getUsersInfo(uids: string[]): Promise<UserInfo[]> {
    try {
      const response = await this.http.get<BlackboxUserInfoData>('/blackbox', {
        params: {
          method: 'userinfo',
          uid: uids.join(','),
          attributes: 27,
          regname: 'yes',
          get_public_name: 'yes',
          userip: '127.0.0.1',
          format: 'json',
        },
      });

      return response.data.users.map(({ id, attributes, display_name }) => ({
        name: attributes['27'],
        avatar: display_name.avatar.default,
        uid: id,
      }));
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }
}
