import qs from 'querystring';

import { Injectable } from '@nestjs/common';
import {
  ConfirmByPasswordInput,
  ConfirmByPasswordPayload,
  ConfirmByPasswordProblem,
  ConfirmByPasswordProblemKind,
} from '@server/graphql-schema';
import { fromModel } from '@server/shared/libs';
import { LoggerService } from '@server/shared/logger';
import { PassportApiResponse, PassportService } from '@server/shared/passport';
import { Context } from '@yandex-int/nest-common';

@Injectable()
export class ConfirmationService {
  constructor(
    private passport: PassportService,
    private logger: LoggerService,
    private context: Context,
  ) {}

  async confirmByPassword(
    input: ConfirmByPasswordInput,
  ): Promise<ConfirmByPasswordProblem | ConfirmByPasswordPayload> {
    const response = await this.post('/1/bundle/auth/password/confirm/commit_password/', {
      track_id: input.trackId,
      password: input.password,
    });

    if (response.data.status === 'error') {
      const { errors } = response.data;
      let reason = ConfirmByPasswordProblemKind.INTERNAL;

      if (errors.includes('captcha.required')) {
        reason = ConfirmByPasswordProblemKind.CAPTCHA_REQUIRED;
      } else if (errors.includes('password.not_matched')) {
        reason = ConfirmByPasswordProblemKind.PASSWORD_NOT_MATCHED;
      } else {
        this.logger.error('ConfirmByPasswordError', { errors, err: errors[0] });
      }

      return fromModel(ConfirmByPasswordProblem, { reason, raw: JSON.stringify(response.data) });
    }

    await this.passport.updateAuthSession(input.trackId);

    return fromModel(ConfirmByPasswordPayload, { successful: true });
  }

  private post<T, U extends Record<string, any> = any>(path: string, data: U) {
    return this.passport.http.post<PassportApiResponse<T>>(
      `${path}?consumer=passport`,
      qs.stringify(data),
      {
        headers: {
          'Ya-Client-Cookie': this.context.req.headers.cookie,
          'Ya-Client-Host': this.context.req.headers.host,
          'Ya-Consumer-Client-Ip': this.context.req.ip,
        },
      },
    );
  }
}
