import axios, { AxiosResponse } from 'axios';
import FormData from 'form-data';

import { Injectable } from '@nestjs/common';
import {
  AddDocumentInput,
  AddDocumentPayload,
  AddDocumentProblem,
  AddDocumentProblemReason,
  BirthCertificateDocument,
  CopyDocumentsFromDiskPayload,
  DeleteDocumentPayload,
  DeleteDocumentProblem,
  DeleteDocumentProblemReason,
  Document,
  DocumentByIdProblem,
  DocumentByIdProblemReason,
  DocumentImage,
  DocumentImageInput,
  DocumentSection,
  DocumentType,
  DriverLicenseDocument,
  InnDocument,
  InternationalIdDocument,
  MedicalInsuranceDocument,
  NationalIdDocument,
  SnilsDocument,
  UpdateDocumentInput,
  UpdateDocumentPayload,
  UpdateDocumentProblem,
  UpdateDocumentProblemReason,
  VehicleRegistrationCertificateDocument,
  VoluntaryMedicalInsuranceDocument,
} from '@server/graphql-schema';
import { BlackboxService } from '@server/shared/blackbox';
import { HttpService } from '@server/shared/http';
import { fromModel } from '@server/shared/libs';

import {
  DocumentType as DataDocumentType,
  DeleteDocumentApiResponse,
  DocumentByIdApiResponse,
  DocumentData,
  DocumentImage as DocumentImageData,
  DocumentsData,
} from './documents.interface';
import { CopyDocumentsFromDiskInput } from './models/copy-documents-from-disk.input';
import { DeleteDocumentInput } from './models/delete-document.input';

@Injectable()
export class DocumentsService {
  constructor(private blackboxService: BlackboxService, private http: HttpService) {}

  async getDocumentById(id: string) {
    const blackbox = await this.blackboxService.getBlackbox();

    try {
      const response = await this.http.get<DocumentByIdApiResponse>('/document/get', {
        params: {
          user_id: blackbox.uid,
          original_id: id,
        },
      });

      const { data } = response;

      if (data.status === 'error') {
        return fromModel(DocumentByIdProblem, {
          reason: DocumentByIdProblemReason.NOT_FOUND,
        });
      }

      const type = this.getDocumentType(data.doc_type);
      const images = data.images.map(this.getImageModel);
      const verificationStatus = data.verification_status?.toLocaleLowerCase();

      if (type === DocumentType.BIRTH_CERTIFICATE) {
        return fromModel(BirthCertificateDocument, {
          id: data.original_id,
          type,
          images,
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          series: data.series,
          docNumber: data.doc_number,
          firstName: data.first_name,
          lastName: data.last_name,
          middleName: data.middle_name,
          birthDate: data.birth_date,
          birthPlace: data.birth_place,
          birthAkt: data.birth_akt,
          fatherLastName: data.father_last_name,
          fatherFirstName: data.father_first_name,
          fatherMiddleName: data.father_middle_name,
          fatherCitizenship: data.father_citizenship,
          fatherBirthDate: data.father_birth_date,
          fatherNationality: data.father_nationality,
          motherLastName: data.mother_last_name,
          motherFirstName: data.mother_first_name,
          motherMiddleName: data.mother_middle_name,
          motherCitizenship: data.mother_citizenship,
          motherBirthDate: data.mother_birth_date,
          motherNationality: data.mother_nationality,
          registrationPlace: data.registration_place,
          issuePlace: data.issue_place,
          issueDate: data.issue_date,
        });
      }

      if (type === DocumentType.DRIVER_LICENSE) {
        return fromModel(DriverLicenseDocument, {
          id: data.original_id,
          type,
          images,
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          lastName: data.last_name,
          lastNameLat: data.last_name_lat,
          firstName: data.first_name,
          firstNameLat: data.first_name_lat,
          middleName: data.middle_name,
          middleNameLat: data.middle_name_lat,
          birthDate: data.birth_date,
          birthPlace: data.birth_place,
          birthPlaceLat: data.birth_place_lat,
          issueDate: data.issue_date,
          endDate: data.end_date,
          issuedBy: data.issued_by,
          issuedByLat: data.issued_by_lat,
          docNumber: data.doc_number,
          issuedCity: data.issued_city,
          issuedCityLat: data.issued_city_lat,
          category: data.category,
          additionalInformation: data.additional_information,
        });
      }

      if (type === DocumentType.INN) {
        return fromModel(InnDocument, {
          id: data.original_id,
          type,
          images,
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          docNumber: data.doc_number,
          firstName: data.first_name,
          lastName: data.last_name,
          middleName: data.middle_name,
          birthDate: data.birth_date,
          gender: data.gender,
          issuedBy: data.issued_by,
          issueDate: data.issue_date,
        });
      }

      if (type === DocumentType.INTERNATIONAL_ID) {
        return fromModel(InternationalIdDocument, {
          id: data.original_id,
          type,
          images,
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          docNumber: data.doc_number,
          lastName: data.last_name,
          lastNameLat: data.last_name_lat,
          firstName: data.first_name,
          firstNameLat: data.first_name_lat,
          citizenship: data.citizenship,
          citizenshipLat: data.citizenship_lat,
          birthDate: data.birth_date,
          gender: data.gender,
          birthPlace: data.birth_place,
          birthPlaceLat: data.birth_place_lat,
          issueDate: data.issue_date,
          endDate: data.end_date,
          issuedBy: data.issued_by,
        });
      }

      if (type === DocumentType.MEDICAL_INSURANCE) {
        return fromModel(MedicalInsuranceDocument, {
          id: data.original_id,
          type,
          images,
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          docNumber: data.doc_number,
          firstName: data.first_name,
          lastName: data.last_name,
          middleName: data.middle_name,
          birthDate: data.birth_date,
          gender: data.gender,
          blank: data.blank,
        });
      }

      if (type === DocumentType.NATIONAL_ID) {
        return fromModel(NationalIdDocument, {
          id: data.original_id,
          type,
          images,
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          docNumber: data.doc_number,
          firstName: data.first_name,
          lastName: data.last_name,
          middleName: data.middle_name,
          birthDate: data.birth_date,
          gender: data.gender,
          birthPlace: data.birth_place,
          issuedBy: data.issued_by,
          issuerSubdivisionCode: data.issuer_subdivision_code,
          issueDate: data.issue_date,
          registrationAddress: data.registration_address_line,
        });
      }

      if (type === DocumentType.SNILS) {
        return fromModel(SnilsDocument, {
          id: data.original_id,
          type,
          images,
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          docNumber: data.doc_number,
          firstName: data.first_name,
          lastName: data.last_name,
          middleName: data.middle_name,
          birthDate: data.birth_date,
          birthPlace: data.birth_place,
          gender: data.gender,
          issueDate: data.issue_date,
        });
      }

      if (type === DocumentType.VOLUNTARY_MEDICAL_INSURANCE) {
        return fromModel(VoluntaryMedicalInsuranceDocument, {
          id: data.original_id,
          type,
          images,
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          docNumber: data.doc_number,
          firstName: data.first_name,
          lastName: data.last_name,
          middleName: data.middle_name,
          birthDate: data.birth_date,
          gender: data.gender,
          issueDate: data.issue_date,
          endDate: data.end_date,
          insurerOrg: data.insurer_org,
        });
      }

      if (type === DocumentType.VEHICLE_REGISTRATION_CERTIFICATE) {
        return fromModel(VehicleRegistrationCertificateDocument, {
          id: data.original_id,
          type,
          images,
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          docNumber: data.doc_number,
          lastName: data.last_name,
          lastNameLat: data.last_name_lat,
          firstName: data.first_name,
          firstNameLat: data.first_name_lat,
          middleName: data.middle_name,
          registrationAddress: data.registration_address,
          issuerSubdivisionCode: data.issuer_subdivision_code,
          issueDate: data.issue_date,
          additionalInformation: data.additional_information,
          carNumber: data.car_number,
          vin: data.vin,
          model: data.model,
          modelLat: data.model_lat,
          vehicleType: data.type,
          category: data.category,
          year: data.year,
          shasiNumber: data.shasi_number,
          bodyNumber: data.body_number,
          color: data.color,
          power: data.power,
          ecoClass: data.eco_class,
          maxWeight: data.max_weight,
          withoutLoadWeight: data.without_load_weight,
          temporaryReg: data.temporary_reg,
          pts: data.pts,
        });
      }

      return fromModel(DocumentByIdProblem, {
        reason: DocumentByIdProblemReason.NOT_FOUND,
      });
    } catch (err) {
      // Если передавать id, которого нет, ручка кинет ошибку
      // Обрабатываем этот случай, иначе прокинем ошибку выше
      if (axios.isAxiosError(err) && err.response?.data?.status === 'error') {
        return fromModel(DocumentByIdProblem, {
          reason: DocumentByIdProblemReason.NOT_FOUND,
        });
      }

      throw err;
    }
  }

  async getDocumentsList(type?: DocumentType) {
    const blackbox = await this.blackboxService.getBlackbox();

    const response = await this.http.get<DocumentsData>('/document/list', {
      params: {
        user_id: blackbox.uid,
        // на бекенде типы это inn, snils, national_id
        // а у нас CONSTANT_CASE
        // либо же делать мапу, либо же toLowerCase, toUpperCase
        doc_type: type ? type.toLowerCase() : undefined,
      },
    });

    const { data } = response;

    const documents = data.documents.map((document) => {
      const type = this.getDocumentType(document.doc_type);
      const images = document.images.map(this.getImageModel);

      if (type === DocumentType.BIRTH_CERTIFICATE) {
        return fromModel(BirthCertificateDocument, {
          id: document.original_id,
          type,
          images,
          verificationStatus: document.verification_status?.toLocaleLowerCase(),
          originalUrl: document.original_url,
          previewUrl: document.preview_url,
          section: this.getDocumentSection(document.doc_type),
          series: document.series,
          docNumber: document.doc_number,
          firstName: document.first_name,
          lastName: document.last_name,
          middleName: document.middle_name,
          birthDate: document.birth_date,
          birthPlace: document.birth_place,
          birthAkt: document.birth_akt,
          fatherLastName: document.father_last_name,
          fatherFirstName: document.father_first_name,
          fatherMiddleName: document.father_middle_name,
          fatherCitizenship: document.father_citizenship,
          fatherBirthDate: document.father_birth_date,
          fatherNationality: document.father_nationality,
          motherLastName: document.mother_last_name,
          motherFirstName: document.mother_first_name,
          motherMiddleName: document.mother_middle_name,
          motherCitizenship: document.mother_citizenship,
          motherBirthDate: document.mother_birth_date,
          motherNationality: document.mother_nationality,
          registrationPlace: document.registration_place,
          issuePlace: document.issue_place,
          issueDate: document.issue_date,
        });
      }

      if (type === DocumentType.DRIVER_LICENSE) {
        return fromModel(DriverLicenseDocument, {
          id: document.original_id,
          type,
          images,
          verificationStatus: document.verification_status?.toLocaleLowerCase(),
          originalUrl: document.original_url,
          previewUrl: document.preview_url,
          section: this.getDocumentSection(document.doc_type),
          lastName: document.last_name,
          lastNameLat: document.last_name_lat,
          firstName: document.first_name,
          firstNameLat: document.first_name_lat,
          middleName: document.middle_name,
          middleNameLat: document.middle_name_lat,
          birthDate: document.birth_date,
          birthPlace: document.birth_place,
          birthPlaceLat: document.birth_place_lat,
          issueDate: document.issue_date,
          endDate: document.end_date,
          issuedBy: document.issued_by,
          issuedByLat: document.issued_by_lat,
          docNumber: document.doc_number,
          issuedCity: document.issued_city,
          issuedCityLat: document.issued_city_lat,
          category: document.category,
          additionalInformation: document.additional_information,
        });
      }

      if (type === DocumentType.INN) {
        return fromModel(InnDocument, {
          id: document.original_id,
          type,
          images,
          verificationStatus: document.verification_status?.toLocaleLowerCase(),
          originalUrl: document.original_url,
          previewUrl: document.preview_url,
          section: this.getDocumentSection(document.doc_type),
          docNumber: document.doc_number,
          firstName: document.first_name,
          lastName: document.last_name,
          middleName: document.middle_name,
          birthDate: document.birth_date,
          gender: document.gender,
          issuedBy: document.issued_by,
          issueDate: document.issue_date,
        });
      }

      if (type === DocumentType.INTERNATIONAL_ID) {
        return fromModel(InternationalIdDocument, {
          id: document.original_id,
          type,
          images,
          verificationStatus: document.verification_status?.toLocaleLowerCase(),
          originalUrl: document.original_url,
          previewUrl: document.preview_url,
          section: this.getDocumentSection(document.doc_type),
          docNumber: document.doc_number,
          lastName: document.last_name,
          lastNameLat: document.last_name_lat,
          firstName: document.first_name,
          firstNameLat: document.first_name_lat,
          citizenship: document.citizenship,
          citizenshipLat: document.citizenship_lat,
          birthDate: document.birth_date,
          gender: document.gender,
          birthPlace: document.birth_place,
          birthPlaceLat: document.birth_place_lat,
          issueDate: document.issue_date,
          endDate: document.end_date,
          issuedBy: document.issued_by,
        });
      }

      if (type === DocumentType.MEDICAL_INSURANCE) {
        return fromModel(MedicalInsuranceDocument, {
          id: document.original_id,
          type,
          images,
          verificationStatus: document.verification_status?.toLocaleLowerCase(),
          originalUrl: document.original_url,
          previewUrl: document.preview_url,
          section: this.getDocumentSection(document.doc_type),
          docNumber: document.doc_number,
          firstName: document.first_name,
          lastName: document.last_name,
          middleName: document.middle_name,
          birthDate: document.birth_date,
          gender: document.gender,
          blank: document.blank,
        });
      }

      if (type === DocumentType.NATIONAL_ID) {
        return fromModel(NationalIdDocument, {
          id: document.original_id,
          type,
          images,
          verificationStatus: document.verification_status?.toLocaleLowerCase(),
          originalUrl: document.original_url,
          previewUrl: document.preview_url,
          section: this.getDocumentSection(document.doc_type),
          docNumber: document.doc_number,
          firstName: document.first_name,
          lastName: document.last_name,
          middleName: document.middle_name,
          birthDate: document.birth_date,
          gender: document.gender,
          birthPlace: document.birth_place,
          issuedBy: document.issued_by,
          issuerSubdivisionCode: document.issuer_subdivision_code,
          issueDate: document.issue_date,
          registrationAddress: document.registration_address_line,
        });
      }

      if (type === DocumentType.SNILS) {
        return fromModel(SnilsDocument, {
          id: document.original_id,
          type,
          images,
          verificationStatus: document.verification_status?.toLocaleLowerCase(),
          originalUrl: document.original_url,
          previewUrl: document.preview_url,
          section: this.getDocumentSection(document.doc_type),
          docNumber: document.doc_number,
          firstName: document.first_name,
          lastName: document.last_name,
          middleName: document.middle_name,
          birthDate: document.birth_date,
          birthPlace: document.birth_place,
          gender: document.gender,
          issueDate: document.issue_date,
        });
      }

      if (type === DocumentType.VOLUNTARY_MEDICAL_INSURANCE) {
        return fromModel(VoluntaryMedicalInsuranceDocument, {
          id: document.original_id,
          type,
          images,
          verificationStatus: document.verification_status?.toLocaleLowerCase(),
          originalUrl: document.original_url,
          previewUrl: document.preview_url,
          section: this.getDocumentSection(document.doc_type),
          docNumber: document.doc_number,
          firstName: document.first_name,
          lastName: document.last_name,
          middleName: document.middle_name,
          birthDate: document.birth_date,
          gender: document.gender,
          issueDate: document.issue_date,
          endDate: document.end_date,
          insurerOrg: document.insurer_org,
        });
      }

      return fromModel(VehicleRegistrationCertificateDocument, {
        id: document.original_id,
        type,
        images,
        verificationStatus: document.verification_status?.toLocaleLowerCase(),
        originalUrl: document.original_url,
        previewUrl: document.preview_url,
        section: this.getDocumentSection(document.doc_type),
        docNumber: document.doc_number,
        lastName: document.last_name,
        lastNameLat: document.last_name_lat,
        firstName: document.first_name,
        firstNameLat: document.first_name_lat,
        middleName: document.middle_name,
        registrationAddress: document.registration_address,
        issuerSubdivisionCode: document.issuer_subdivision_code,
        issueDate: document.issue_date,
        additionalInformation: document.additional_information,
        carNumber: document.car_number,
        vin: document.vin,
        model: document.model,
        modelLat: document.model_lat,
        category: document.category,
        vehicleType: document.type,
        year: document.year,
        shasiNumber: document.shasi_number,
        bodyNumber: document.body_number,
        color: document.color,
        power: document.power,
        ecoClass: document.eco_class,
        maxWeight: document.max_weight,
        withoutLoadWeight: document.without_load_weight,
        temporaryReg: document.temporary_reg,
        pts: document.pts,
      });
    });

    return documents;
  }

  async addDocument(input: AddDocumentInput) {
    const blackbox = await this.blackboxService.getBlackbox();

    if (input.type === DocumentType.BIRTH_CERTIFICATE && input.birthCertificateInput) {
      const images = input.birthCertificateInput.images.map<DocumentImageData>(this.getImageData);

      await this.http.post<DocumentsData>(`/document/add?user_id=${blackbox.uid}`, {
        doc_type: input.type.toLocaleLowerCase(),
        images,
        series: input.birthCertificateInput.series,
        doc_number: input.birthCertificateInput.docNumber,
        first_name: input.birthCertificateInput.firstName,
        last_name: input.birthCertificateInput.lastName,
        middle_name: input.birthCertificateInput.middleName,
        birth_data: input.birthCertificateInput.birthDate,
        birth_place: input.birthCertificateInput.birthPlace,
        birth_akt: input.birthCertificateInput.birthAkt,
        father_last_name: input.birthCertificateInput.fatherLastName,
        father_first_name: input.birthCertificateInput.fatherFirstName,
        father_middle_name: input.birthCertificateInput.fatherMiddleName,
        father_citizenship: input.birthCertificateInput.fatherCitizenship,
        father_birth_date: input.birthCertificateInput.fatherBirthDate,
        father_nationality: input.birthCertificateInput.fatherNationality,
        mother_last_name: input.birthCertificateInput.motherLastName,
        mother_first_name: input.birthCertificateInput.motherFirstName,
        mother_middle_name: input.birthCertificateInput.motherMiddleName,
        mother_citizenship: input.birthCertificateInput.motherCitizenship,
        mother_birth_date: input.birthCertificateInput.motherBirthDate,
        mother_nationality: input.birthCertificateInput.motherNationality,
        registration_place: input.birthCertificateInput.registrationPlace,
        issue_place: input.birthCertificateInput.issuePlace,
        issue_date: input.birthCertificateInput.issueDate,
      });

      return fromModel(AddDocumentPayload, {});
    }

    if (input.type === DocumentType.DRIVER_LICENSE && input.driverLicenseInput) {
      const images = input.driverLicenseInput.images.map<DocumentImageData>(this.getImageData);

      await this.http.post<DocumentsData>(`/document/add?user_id=${blackbox.uid}`, {
        doc_type: input.type.toLocaleLowerCase(),
        images,
        last_name: input.driverLicenseInput.lastName,
        last_name_lat: input.driverLicenseInput.lastNameLat,
        first_name: input.driverLicenseInput.firstName,
        first_name_lat: input.driverLicenseInput.firstNameLat,
        middle_name: input.driverLicenseInput.middleName,
        middle_name_lat: input.driverLicenseInput.middleNameLat,
        birth_date: input.driverLicenseInput.birthDate,
        birth_place: input.driverLicenseInput.birthPlace,
        birth_place_lat: input.driverLicenseInput.birthPlaceLat,
        issue_date: input.driverLicenseInput.issueDate,
        end_date: input.driverLicenseInput.endDate,
        issued_by: input.driverLicenseInput.issuedBy,
        issued_by_lat: input.driverLicenseInput.issuedByLat,
        doc_number: input.driverLicenseInput.docNumber,
        issued_city: input.driverLicenseInput.issuedCity,
        issued_city_lat: input.driverLicenseInput.issuedCityLat,
        category: input.driverLicenseInput.category,
        additional_information: input.driverLicenseInput.additionalInformation,
      });

      return fromModel(AddDocumentPayload, {});
    }

    if (input.type === DocumentType.INN && input.innInput) {
      const images = input.innInput.images.map<DocumentImageData>(this.getImageData);

      await this.http.post<DocumentsData>(`/document/add?user_id=${blackbox.uid}`, {
        doc_type: input.type.toLocaleLowerCase(),
        images,
        service: 'passport',
        doc_number: input.innInput.docNumber,
        first_name: input.innInput.firstName,
        last_name: input.innInput.lastName,
        middle_name: input.innInput.middleName,
        birth_date: input.innInput.birthDate,
        gender: input.innInput.gender,
        issued_by: input.innInput.issuedBy,
        issue_date: input.innInput.issueDate,
      });

      return fromModel(AddDocumentPayload, {});
    }

    if (input.type === DocumentType.INTERNATIONAL_ID && input.internationalIdInput) {
      const images = input.internationalIdInput.images.map<DocumentImageData>(this.getImageData);

      await this.http.post<DocumentsData>(`/document/add?user_id=${blackbox.uid}`, {
        doc_type: input.type.toLocaleLowerCase(),
        images,
        service: 'passport',
        doc_number: input.internationalIdInput.docNumber,
        last_name: input.internationalIdInput.lastName,
        last_name_lat: input.internationalIdInput.lastNameLat,
        first_name: input.internationalIdInput.firstName,
        first_name_lat: input.internationalIdInput.firstNameLat,
        citizenship: input.internationalIdInput.citizenship,
        citizenship_lat: input.internationalIdInput.citizenshipLat,
        birth_date: input.internationalIdInput.birthDate,
        gender: input.internationalIdInput.gender,
        birth_place: input.internationalIdInput.birthPlace,
        birth_place_lat: input.internationalIdInput.birthPlaceLat,
        issue_date: input.internationalIdInput.issueDate,
        end_date: input.internationalIdInput.endDate,
        issued_by: input.internationalIdInput.issuedBy,
      });

      return fromModel(AddDocumentPayload, {});
    }

    if (input.type === DocumentType.MEDICAL_INSURANCE && input.medicalInsuranceInput) {
      const images = input.medicalInsuranceInput.images.map<DocumentImageData>(this.getImageData);

      await this.http.post<DocumentsData>(`/document/add?user_id=${blackbox.uid}`, {
        doc_type: input.type.toLocaleLowerCase(),
        images,
        doc_number: input.medicalInsuranceInput.docNumber,
        first_name: input.medicalInsuranceInput.firstName,
        last_name: input.medicalInsuranceInput.lastName,
        middle_name: input.medicalInsuranceInput.middleName,
        birth_date: input.medicalInsuranceInput.birthDate,
        gender: input.medicalInsuranceInput.gender,
        blank: input.medicalInsuranceInput.blank,
      });

      return fromModel(AddDocumentPayload, {});
    }

    if (input.type === DocumentType.NATIONAL_ID && input.nationalIdInput) {
      const images = input.nationalIdInput.images.map<DocumentImageData>(this.getImageData);

      await this.http.post<DocumentsData>(`/document/add?user_id=${blackbox.uid}`, {
        doc_type: input.type.toLowerCase(),
        images,
        doc_number: input.nationalIdInput.docNumber,
        first_name: input.nationalIdInput.firstName,
        last_name: input.nationalIdInput.lastName,
        middle_name: input.nationalIdInput.middleName,
        birth_date: input.nationalIdInput.birthDate,
        gender: input.nationalIdInput.gender,
        birth_place: input.nationalIdInput.birthPlace,
        issued_by: input.nationalIdInput.issuedBy,
        issuer_subdivision_code: input.nationalIdInput.issuerSubdivisionCode,
        issue_date: input.nationalIdInput.issueDate,
        registration_address_line: input.nationalIdInput.registrationAddress,
      });

      return fromModel(AddDocumentPayload, {});
    }

    if (input.type === DocumentType.SNILS && input.snilsInput) {
      const images = input.snilsInput.images.map<DocumentImageData>(this.getImageData);

      await this.http.post<DocumentsData>(`/document/add?user_id=${blackbox.uid}`, {
        doc_type: input.type.toLocaleLowerCase(),
        images,
        doc_number: input.snilsInput.docNumber,
        first_name: input.snilsInput.firstName,
        last_name: input.snilsInput.lastName,
        middle_name: input.snilsInput.middleName,
        birth_date: input.snilsInput.birthDate,
        birth_place: input.snilsInput.birthPlace,
        gender: input.snilsInput.gender,
        issue_date: input.snilsInput.issueDate,
      });

      return fromModel(AddDocumentPayload, {});
    }

    if (
      input.type === DocumentType.VOLUNTARY_MEDICAL_INSURANCE &&
      input.voluntaryMedicalInsuranceInput
    ) {
      const images = input.voluntaryMedicalInsuranceInput.images.map<DocumentImageData>(
        this.getImageData,
      );

      await this.http.post<DocumentsData>(`/document/add?user_id=${blackbox.uid}`, {
        doc_type: input.type.toLocaleLowerCase(),
        images,
        doc_number: input.voluntaryMedicalInsuranceInput.docNumber,
        first_name: input.voluntaryMedicalInsuranceInput.firstName,
        last_name: input.voluntaryMedicalInsuranceInput.lastName,
        middle_name: input.voluntaryMedicalInsuranceInput.middleName,
        birth_date: input.voluntaryMedicalInsuranceInput.birthDate,
        gender: input.voluntaryMedicalInsuranceInput.gender,
        issue_date: input.voluntaryMedicalInsuranceInput.issueDate,
        end_date: input.voluntaryMedicalInsuranceInput.endDate,
        insurer_org: input.voluntaryMedicalInsuranceInput.insurerOrg,
      });

      return fromModel(AddDocumentPayload, {});
    }

    if (
      input.type === DocumentType.VEHICLE_REGISTRATION_CERTIFICATE &&
      input.vehicleRegistrationCertificateInput
    ) {
      const images = input.vehicleRegistrationCertificateInput.images.map<DocumentImageData>(
        this.getImageData,
      );

      await this.http.post<DocumentsData>(`/document/add?user_id=${blackbox.uid}`, {
        doc_type: input.type.toLocaleLowerCase(),
        images,
        doc_number: input.vehicleRegistrationCertificateInput.docNumber,
        last_name: input.vehicleRegistrationCertificateInput.lastName,
        last_name_lat: input.vehicleRegistrationCertificateInput.lastNameLat,
        first_name: input.vehicleRegistrationCertificateInput.firstName,
        first_name_lat: input.vehicleRegistrationCertificateInput.firstNameLat,
        middle_name: input.vehicleRegistrationCertificateInput.middleName,
        registration_address: input.vehicleRegistrationCertificateInput.registrationAddress,
        issuer_subdivision_code: input.vehicleRegistrationCertificateInput.issuerSubdivisionCode,
        issue_date: input.vehicleRegistrationCertificateInput.issueDate,
        additional_information: input.vehicleRegistrationCertificateInput.additionalInformation,
        car_number: input.vehicleRegistrationCertificateInput.carNumber,
        vin: input.vehicleRegistrationCertificateInput.vin,
        model: input.vehicleRegistrationCertificateInput.model,
        model_lat: input.vehicleRegistrationCertificateInput.modelLat,
        type: input.vehicleRegistrationCertificateInput.vehicleType,
        category: input.vehicleRegistrationCertificateInput.category,
        year: input.vehicleRegistrationCertificateInput.year,
        shasi_number: input.vehicleRegistrationCertificateInput.shasiNumber,
        body_number: input.vehicleRegistrationCertificateInput.bodyNumber,
        color: input.vehicleRegistrationCertificateInput.color,
        power: input.vehicleRegistrationCertificateInput.power,
        eco_class: input.vehicleRegistrationCertificateInput.ecoClass,
        max_weight: input.vehicleRegistrationCertificateInput.maxWeight,
        without_load_weight: input.vehicleRegistrationCertificateInput.withoutLoadWeight,
        temporary_reg: input.vehicleRegistrationCertificateInput.temporaryReg,
        pts: input.vehicleRegistrationCertificateInput.pts,
      });

      return fromModel(AddDocumentPayload, {});
    }

    return fromModel(AddDocumentProblem, {
      reason: AddDocumentProblemReason.INTERNAL,
    });
  }

  async updateDocument(input: UpdateDocumentInput) {
    const blackbox = await this.blackboxService.getBlackbox();

    if (input.type === DocumentType.BIRTH_CERTIFICATE && input.birthCertificateInput) {
      const images = input.birthCertificateInput.images.map<DocumentImageData>(this.getImageData);

      await this.http.post<DocumentsData>(`/document/update?user_id=${blackbox.uid}`, {
        original_id: input.birthCertificateInput.id,
        doc_type: input.type.toLocaleLowerCase(),
        verification_status: input.birthCertificateInput.verificationStatus,
        images,
        service: 'passport',
        series: input.birthCertificateInput.series,
        doc_number: input.birthCertificateInput.docNumber,
        first_name: input.birthCertificateInput.firstName,
        last_name: input.birthCertificateInput.lastName,
        middle_name: input.birthCertificateInput.middleName,
        birth_date: input.birthCertificateInput.birthDate,
        birth_place: input.birthCertificateInput.birthPlace,
        birth_akt: input.birthCertificateInput.birthAkt,
        father_last_name: input.birthCertificateInput.fatherLastName,
        father_first_name: input.birthCertificateInput.fatherFirstName,
        father_middle_name: input.birthCertificateInput.fatherMiddleName,
        father_citizenship: input.birthCertificateInput.fatherCitizenship,
        father_birth_date: input.birthCertificateInput.fatherBirthDate,
        father_nationality: input.birthCertificateInput.fatherNationality,
        mother_last_name: input.birthCertificateInput.motherLastName,
        mother_first_name: input.birthCertificateInput.motherFirstName,
        mother_middle_name: input.birthCertificateInput.motherMiddleName,
        mother_citizenship: input.birthCertificateInput.motherCitizenship,
        mother_birth_date: input.birthCertificateInput.motherBirthDate,
        mother_nationality: input.birthCertificateInput.motherNationality,
        registration_place: input.birthCertificateInput.registrationPlace,
        issue_place: input.birthCertificateInput.issuePlace,
        issue_date: input.birthCertificateInput.issueDate,
      });

      return fromModel(UpdateDocumentPayload, {});
    }

    if (input.type === DocumentType.DRIVER_LICENSE && input.driverLicenseInput) {
      const images = input.driverLicenseInput.images.map<DocumentImageData>(this.getImageData);

      await this.http.post<DocumentsData>(`/document/update?user_id=${blackbox.uid}`, {
        original_id: input.driverLicenseInput.id,
        verification_status: input.driverLicenseInput.verificationStatus,
        doc_type: input.type.toLocaleLowerCase(),
        images,
        service: 'passport',
        last_name: input.driverLicenseInput.lastName,
        last_name_lat: input.driverLicenseInput.lastNameLat,
        first_name: input.driverLicenseInput.firstName,
        first_name_lat: input.driverLicenseInput.firstNameLat,
        middle_name: input.driverLicenseInput.middleName,
        middle_name_lat: input.driverLicenseInput.middleNameLat,
        birth_date: input.driverLicenseInput.birthDate,
        birth_place: input.driverLicenseInput.birthPlace,
        birth_place_lat: input.driverLicenseInput.birthPlaceLat,
        issue_date: input.driverLicenseInput.issueDate,
        end_date: input.driverLicenseInput.endDate,
        issued_by: input.driverLicenseInput.issuedBy,
        issued_by_lat: input.driverLicenseInput.issuedByLat,
        doc_number: input.driverLicenseInput.docNumber,
        issued_city: input.driverLicenseInput.issuedCity,
        issued_city_lat: input.driverLicenseInput.issuedCityLat,
        category: input.driverLicenseInput.category,
        additional_information: input.driverLicenseInput.additionalInformation,
      });

      return fromModel(UpdateDocumentPayload, {});
    }

    if (input.type === DocumentType.INN && input.innInput) {
      const images = input.innInput.images.map<DocumentImageData>(this.getImageData);

      await this.http.post<DocumentsData>(`/document/update?user_id=${blackbox.uid}`, {
        original_id: input.innInput.id,
        verification_status: input.innInput.verificationStatus,
        doc_type: input.type.toLocaleLowerCase(),
        images,
        service: 'passport',
        doc_number: input.innInput.docNumber,
        first_name: input.innInput.firstName,
        last_name: input.innInput.lastName,
        middle_name: input.innInput.middleName,
        birth_date: input.innInput.birthDate,
        gender: input.innInput.gender,
        issued_by: input.innInput.issuedBy,
        issue_date: input.innInput.issueDate,
      });

      return fromModel(UpdateDocumentPayload, {});
    }

    if (input.type === DocumentType.INTERNATIONAL_ID && input.internationalIdInput) {
      const images = input.internationalIdInput.images.map<DocumentImageData>(this.getImageData);

      await this.http.post<DocumentsData>(`/document/update?user_id=${blackbox.uid}`, {
        original_id: input.internationalIdInput.id,
        verification_status: input.internationalIdInput.verificationStatus,
        doc_type: input.type.toLocaleLowerCase(),
        images,
        service: 'passport',
        doc_number: input.internationalIdInput.docNumber,
        last_name: input.internationalIdInput.lastName,
        last_name_lat: input.internationalIdInput.lastNameLat,
        first_name: input.internationalIdInput.firstName,
        first_name_lat: input.internationalIdInput.firstNameLat,
        citizenship: input.internationalIdInput.citizenship,
        citizenship_lat: input.internationalIdInput.citizenshipLat,
        birth_date: input.internationalIdInput.birthDate,
        gender: input.internationalIdInput.gender,
        birth_place: input.internationalIdInput.birthPlace,
        birth_place_lat: input.internationalIdInput.birthPlaceLat,
        issue_date: input.internationalIdInput.issueDate,
        end_date: input.internationalIdInput.endDate,
        issued_by: input.internationalIdInput.issuedBy,
      });

      return fromModel(UpdateDocumentPayload, {});
    }

    if (input.type === DocumentType.MEDICAL_INSURANCE && input.medicalInsuranceInput) {
      const images = input.medicalInsuranceInput.images.map<DocumentImageData>(this.getImageData);

      await this.http.post<DocumentsData>(`/document/update?user_id=${blackbox.uid}`, {
        original_id: input.medicalInsuranceInput.id,
        verification_status: input.medicalInsuranceInput.verificationStatus,
        doc_type: input.type.toLocaleLowerCase(),
        images,
        service: 'passport',
        doc_number: input.medicalInsuranceInput.docNumber,
        first_name: input.medicalInsuranceInput.firstName,
        last_name: input.medicalInsuranceInput.lastName,
        middle_name: input.medicalInsuranceInput.middleName,
        birth_date: input.medicalInsuranceInput.birthDate,
        gender: input.medicalInsuranceInput.gender,
        blank: input.medicalInsuranceInput.blank,
      });

      return fromModel(UpdateDocumentPayload, {});
    }

    if (input.type === DocumentType.NATIONAL_ID && input.nationalIdInput) {
      const images = input.nationalIdInput.images.map<DocumentImageData>(this.getImageData);

      await this.http.post<DocumentsData>(`/document/update?user_id=${blackbox.uid}`, {
        original_id: input.nationalIdInput.id,
        verification_status: input.nationalIdInput.verificationStatus,
        doc_type: input.type.toLocaleLowerCase(),
        images,
        service: 'passport',
        doc_number: input.nationalIdInput.docNumber,
        first_name: input.nationalIdInput.firstName,
        last_name: input.nationalIdInput.lastName,
        middle_name: input.nationalIdInput.middleName,
        birth_date: input.nationalIdInput.birthDate,
        gender: input.nationalIdInput.gender,
        birth_place: input.nationalIdInput.birthPlace,
        issued_by: input.nationalIdInput.issuedBy,
        issuer_subdivision_code: input.nationalIdInput.issuerSubdivisionCode,
        issue_date: input.nationalIdInput.issueDate,
        registration_address_line: input.nationalIdInput.registrationAddress,
      });

      return fromModel(UpdateDocumentPayload, {});
    }

    if (input.type === DocumentType.SNILS && input.snilsInput) {
      const images = input.snilsInput.images.map<DocumentImageData>(this.getImageData);

      await this.http.post<DocumentsData>(`/document/update?user_id=${blackbox.uid}`, {
        original_id: input.snilsInput.id,
        verification_status: input.snilsInput.verificationStatus,
        doc_type: input.type.toLocaleLowerCase(),
        images,
        service: 'passport',
        doc_number: input.snilsInput.docNumber,
        first_name: input.snilsInput.firstName,
        last_name: input.snilsInput.lastName,
        middle_name: input.snilsInput.middleName,
        birth_date: input.snilsInput.birthDate,
        birth_place: input.snilsInput.birthPlace,
        gender: input.snilsInput.gender,
        issue_date: input.snilsInput.issueDate,
      });

      return fromModel(UpdateDocumentPayload, {});
    }

    if (
      input.type === DocumentType.VOLUNTARY_MEDICAL_INSURANCE &&
      input.voluntaryMedicalInsuranceInput
    ) {
      const images = input.voluntaryMedicalInsuranceInput.images.map<DocumentImageData>(
        this.getImageData,
      );

      await this.http.post<DocumentsData>(`/document/update?user_id=${blackbox.uid}`, {
        original_id: input.voluntaryMedicalInsuranceInput.id,
        verification_status: input.voluntaryMedicalInsuranceInput.verificationStatus,
        doc_type: input.type.toLocaleLowerCase(),
        images,
        service: 'passport',
        doc_number: input.voluntaryMedicalInsuranceInput.docNumber,
        first_name: input.voluntaryMedicalInsuranceInput.firstName,
        last_name: input.voluntaryMedicalInsuranceInput.lastName,
        middle_name: input.voluntaryMedicalInsuranceInput.middleName,
        birth_date: input.voluntaryMedicalInsuranceInput.birthDate,
        gender: input.voluntaryMedicalInsuranceInput.gender,
        issue_date: input.voluntaryMedicalInsuranceInput.issueDate,
        end_date: input.voluntaryMedicalInsuranceInput.endDate,
        insurer_org: input.voluntaryMedicalInsuranceInput.insurerOrg,
      });

      return fromModel(UpdateDocumentPayload, {});
    }

    if (
      input.type === DocumentType.VEHICLE_REGISTRATION_CERTIFICATE &&
      input.vehicleRegistrationCertificateInput
    ) {
      const images = input.vehicleRegistrationCertificateInput.images.map<DocumentImageData>(
        this.getImageData,
      );

      await this.http.post<DocumentsData>(`/document/update?user_id=${blackbox.uid}`, {
        original_id: input.vehicleRegistrationCertificateInput.id,
        verification_status: input.vehicleRegistrationCertificateInput.verificationStatus,
        doc_type: input.type.toLocaleLowerCase(),
        images,
        service: 'passport',
        doc_number: input.vehicleRegistrationCertificateInput.docNumber,
        last_name: input.vehicleRegistrationCertificateInput.lastName,
        last_name_lat: input.vehicleRegistrationCertificateInput.lastNameLat,
        first_name: input.vehicleRegistrationCertificateInput.firstName,
        first_name_lat: input.vehicleRegistrationCertificateInput.firstNameLat,
        middle_name: input.vehicleRegistrationCertificateInput.middleName,
        registration_address: input.vehicleRegistrationCertificateInput.registrationAddress,
        issuer_subdivision_code: input.vehicleRegistrationCertificateInput.issuerSubdivisionCode,
        issue_date: input.vehicleRegistrationCertificateInput.issueDate,
        additional_information: input.vehicleRegistrationCertificateInput.additionalInformation,
        car_number: input.vehicleRegistrationCertificateInput.carNumber,
        vin: input.vehicleRegistrationCertificateInput.vin,
        model: input.vehicleRegistrationCertificateInput.model,
        model_lat: input.vehicleRegistrationCertificateInput.modelLat,
        type: input.vehicleRegistrationCertificateInput.vehicleType,
        category: input.vehicleRegistrationCertificateInput.category,
        year: input.vehicleRegistrationCertificateInput.year,
        shasi_number: input.vehicleRegistrationCertificateInput.shasiNumber,
        body_number: input.vehicleRegistrationCertificateInput.bodyNumber,
        color: input.vehicleRegistrationCertificateInput.color,
        power: input.vehicleRegistrationCertificateInput.power,
        eco_class: input.vehicleRegistrationCertificateInput.ecoClass,
        max_weight: input.vehicleRegistrationCertificateInput.maxWeight,
        without_load_weight: input.vehicleRegistrationCertificateInput.withoutLoadWeight,
        temporary_reg: input.vehicleRegistrationCertificateInput.temporaryReg,
        pts: input.vehicleRegistrationCertificateInput.pts,
      });

      return fromModel(UpdateDocumentPayload, {});
    }

    return fromModel(UpdateDocumentProblem, {
      reason: UpdateDocumentProblemReason.INTERNAL,
    });
  }

  async deleteDocument(input: DeleteDocumentInput) {
    const blackbox = await this.blackboxService.getBlackbox();

    const response = await this.http.get<DeleteDocumentApiResponse>('/document/delete', {
      params: {
        user_id: blackbox.uid,
        original_id: input.id,
      },
    });

    if (response.data.status === 'error') {
      return fromModel(DeleteDocumentProblem, {
        reason: DeleteDocumentProblemReason.INTERNAL,
      });
    }

    return fromModel(DeleteDocumentPayload, {});
  }

  async copyDocumentsFromDisk(input: CopyDocumentsFromDiskInput, type: DocumentType) {
    const blackbox = await this.blackboxService.getBlackbox();

    const requests = input.paths.reduce<Promise<AxiosResponse<DocumentData, any>>[]>(
      (acc, path) => {
        acc.push(
          this.http.get<DocumentData>('/document/image/disk/copy', {
            params: {
              user_id: blackbox.uid,
              path,
              doc_type: type.toLowerCase(),
            },
          }),
        );

        return acc;
      },
      [],
    );

    await Promise.all(requests);

    return fromModel(CopyDocumentsFromDiskPayload, {});
  }

  async uploadImage(
    file: Express.Multer.File,
    type: DocumentType,
    id?: string,
    save?: 'true' | 'false',
  ): Promise<DocumentImage | Document> {
    const blackbox = await this.blackboxService.getBlackbox();

    const fd = new FormData();
    const searchParams = new URLSearchParams();

    fd.append('file', file.buffer, file.originalname);
    searchParams.set('user_id', blackbox.uid ?? '');
    searchParams.set('save', save || 'true');

    if (id) {
      searchParams.set('original_id', id);
    }

    // ручка может возращать новую картинку, если мы ее не сохраняем
    // или же весь новый документ, если мы картинку сохраняем
    const response = await this.http.post<DocumentImageData | DocumentData>(
      `/document/image/upload?${searchParams.toString()}`,
      fd,
      {
        headers: fd.getHeaders(),
      },
    );

    const { data } = response;

    // при помощи id проверяем, что к нам пришло - документ или картинка
    // поле id есть только у документа
    if ('id' in data) {
      const images = data.images.map(this.getImageModel);
      const verificationStatus = data.verification_status?.toLocaleLowerCase();

      if (data.doc_type === DataDocumentType.BIRTH_CERTIFICATE) {
        return fromModel(BirthCertificateDocument, {
          id: data.original_id,
          type: this.getDocumentType(data.doc_type),
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          series: data.series,
          docNumber: data.doc_number,
          firstName: data.first_name,
          lastName: data.last_name,
          middleName: data.middle_name,
          birthDate: data.birth_date,
          birthPlace: data.birth_place,
          birthAkt: data.birth_akt,
          fatherLastName: data.father_last_name,
          fatherFirstName: data.father_first_name,
          fatherMiddleName: data.father_middle_name,
          fatherCitizenship: data.father_citizenship,
          fatherBirthDate: data.father_birth_date,
          fatherNationality: data.father_nationality,
          motherLastName: data.mother_last_name,
          motherFirstName: data.mother_first_name,
          motherMiddleName: data.mother_middle_name,
          motherCitizenship: data.mother_citizenship,
          motherBirthDate: data.mother_birth_date,
          motherNationality: data.mother_nationality,
          registrationPlace: data.registration_place,
          issuePlace: data.issue_place,
          issueDate: data.issue_date,
          images,
        });
      }

      if (data.doc_type === DataDocumentType.DRIVER_LICENSE) {
        return fromModel(DriverLicenseDocument, {
          id: data.original_id,
          type: this.getDocumentType(data.doc_type),
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          lastName: data.last_name,
          lastNameLat: data.last_name_lat,
          firstName: data.first_name,
          firstNameLat: data.first_name_lat,
          middleName: data.middle_name,
          middleNameLat: data.middle_name_lat,
          birthDate: data.birth_date,
          birthPlace: data.birth_place,
          birthPlaceLat: data.birth_place_lat,
          issueDate: data.issue_date,
          endDate: data.end_date,
          issuedBy: data.issued_by,
          issuedByLat: data.issued_by_lat,
          docNumber: data.doc_number,
          issuedCity: data.issued_city,
          issuedCityLat: data.issued_city_lat,
          category: data.category,
          additionalInformation: data.additional_information,
          images,
        });
      }

      if (data.doc_type === DataDocumentType.INN) {
        return fromModel(InnDocument, {
          id: data.original_id,
          type: this.getDocumentType(data.doc_type),
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          docNumber: data.doc_number,
          firstName: data.first_name,
          lastName: data.last_name,
          middleName: data.middle_name,
          birthDate: data.birth_date,
          gender: data.gender,
          issuedBy: data.issued_by,
          issueDate: data.issue_date,
          images,
        });
      }

      if (data.doc_type === DataDocumentType.INTERNATIONAL_ID) {
        return fromModel(InternationalIdDocument, {
          id: data.original_id,
          type: this.getDocumentType(data.doc_type),
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          images,
          docNumber: data.doc_number,
          lastName: data.last_name,
          lastNameLat: data.last_name_lat,
          firstName: data.first_name,
          firstNameLat: data.first_name_lat,
          citizenship: data.citizenship,
          citizenshipLat: data.citizenship_lat,
          birthDate: data.birth_date,
          gender: data.gender,
          birthPlace: data.birth_place,
          birthPlaceLat: data.birth_place_lat,
          issueDate: data.issue_date,
          endDate: data.end_date,
          issuedBy: data.issued_by,
        });
      }

      if (data.doc_type === DataDocumentType.MEDICAL_INSURANCE) {
        return fromModel(MedicalInsuranceDocument, {
          id: data.original_id,
          type: this.getDocumentType(data.doc_type),
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          docNumber: data.doc_number,
          firstName: data.first_name,
          lastName: data.last_name,
          middleName: data.middle_name,
          birthDate: data.birth_date,
          gender: data.gender,
          blank: data.blank,
          images,
        });
      }

      if (data.doc_type === DataDocumentType.NATIONAL_ID) {
        return fromModel(NationalIdDocument, {
          id: data.original_id,
          type: this.getDocumentType(data.doc_type),
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          docNumber: data.doc_number,
          firstName: data.first_name,
          lastName: data.last_name,
          middleName: data.middle_name,
          birthDate: data.birth_date,
          gender: data.gender,
          birthPlace: data.birth_place,
          issuedBy: data.issued_by,
          issuerSubdivisionCode: data.issuer_subdivision_code,
          issueDate: data.issue_date,
          registrationAddress: data.registration_address_line,
          images,
        });
      }

      if (data.doc_type === DataDocumentType.SNILS) {
        return fromModel(SnilsDocument, {
          id: data.original_id,
          type: this.getDocumentType(data.doc_type),
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          docNumber: data.doc_number,
          firstName: data.first_name,
          lastName: data.last_name,
          middleName: data.middle_name,
          birthDate: data.birth_date,
          birthPlace: data.birth_place,
          gender: data.gender,
          issueDate: data.issue_date,
          images,
        });
      }

      if (data.doc_type === DataDocumentType.VOLUNTARY_MEDICAL_INSURANCE) {
        return fromModel(VoluntaryMedicalInsuranceDocument, {
          id: data.original_id,
          type: this.getDocumentType(data.doc_type),
          verificationStatus,
          originalUrl: data.original_url,
          previewUrl: data.preview_url,
          section: this.getDocumentSection(data.doc_type),
          docNumber: data.doc_number,
          firstName: data.first_name,
          lastName: data.last_name,
          middleName: data.middle_name,
          birthDate: data.birth_date,
          gender: data.gender,
          issueDate: data.issue_date,
          endDate: data.end_date,
          insurerOrg: data.insurer_org,
          images,
        });
      }

      return fromModel(VehicleRegistrationCertificateDocument, {
        id: data.original_id,
        type: this.getDocumentType(data.doc_type),
        verificationStatus,
        originalUrl: data.original_url,
        previewUrl: data.preview_url,
        section: this.getDocumentSection(data.doc_type),
        docNumber: data.doc_number,
        lastName: data.last_name,
        lastNameLat: data.last_name_lat,
        firstName: data.first_name,
        firstNameLat: data.first_name_lat,
        middleName: data.middle_name,
        registrationAddress: data.registration_address,
        issuerSubdivisionCode: data.issuer_subdivision_code,
        issueDate: data.issue_date,
        additionalInformation: data.additional_information,
        carNumber: data.car_number,
        vin: data.vin,
        model: data.model,
        modelLat: data.model_lat,
        vehicleType: data.type,
        category: data.category,
        year: data.year,
        shasiNumber: data.shasi_number,
        bodyNumber: data.body_number,
        color: data.color,
        power: data.power,
        ecoClass: data.eco_class,
        maxWeight: data.max_weight,
        withoutLoadWeight: data.without_load_weight,
        temporaryReg: data.temporary_reg,
        pts: data.pts,
        images,
      });
    }

    return this.getImageModel(data);
  }

  private getDocumentType(documentType: DataDocumentType): DocumentType {
    switch (documentType) {
      case DataDocumentType.BIRTH_CERTIFICATE:
        return DocumentType.BIRTH_CERTIFICATE;
      case DataDocumentType.DRIVER_LICENSE:
        return DocumentType.DRIVER_LICENSE;
      case DataDocumentType.INN:
        return DocumentType.INN;
      case DataDocumentType.INTERNATIONAL_ID:
        return DocumentType.INTERNATIONAL_ID;
      case DataDocumentType.MEDICAL_INSURANCE:
        return DocumentType.MEDICAL_INSURANCE;
      case DataDocumentType.NATIONAL_ID:
        return DocumentType.NATIONAL_ID;
      case DataDocumentType.SNILS:
        return DocumentType.SNILS;
      case DataDocumentType.VEHICLE_REGISTRATION_CERTIFICATE:
        return DocumentType.VEHICLE_REGISTRATION_CERTIFICATE;
      case DataDocumentType.VOLUNTARY_MEDICAL_INSURANCE:
        return DocumentType.VOLUNTARY_MEDICAL_INSURANCE;
    }
  }

  private getDocumentSection(documentType: DataDocumentType): DocumentSection {
    switch (documentType) {
      case DataDocumentType.BIRTH_CERTIFICATE:
        return DocumentSection.PERSONAL;
      case DataDocumentType.DRIVER_LICENSE:
        return DocumentSection.TRANSPORT;
      case DataDocumentType.INN:
        return DocumentSection.FINANCE;
      case DataDocumentType.INTERNATIONAL_ID:
        return DocumentSection.PERSONAL;
      case DataDocumentType.MEDICAL_INSURANCE:
        return DocumentSection.HEALTH;
      case DataDocumentType.NATIONAL_ID:
        return DocumentSection.PERSONAL;
      case DataDocumentType.SNILS:
        return DocumentSection.FINANCE;
      case DataDocumentType.VEHICLE_REGISTRATION_CERTIFICATE:
        return DocumentSection.TRANSPORT;
      case DataDocumentType.VOLUNTARY_MEDICAL_INSURANCE:
        return DocumentSection.HEALTH;
    }
  }

  private getImageModel(image: DocumentImageData) {
    return fromModel(DocumentImage, {
      id: image.image_id,
      originalUri: image.original_uri,
      originalUrl: image.original_url,
      originalWidth: image.original_width,
      originalHeight: image.original_height,
      previewUri: image.preview_uri,
      previewUrl: image.preview_url,
      previewWidth: image.preview_width,
      previewHeight: image.preview_height,
    });
  }

  private getImageData(image: DocumentImageInput): DocumentImageData {
    return {
      image_id: image.id,
      original_uri: image.originalUri ?? '',
      original_url: image.originalUrl,
      original_width: image.originalWidth ?? undefined,
      original_height: image.previewHeight ?? undefined,
      preview_uri: image.previewUri ?? '',
      preview_url: image.previewUrl ?? '',
      preview_width: image.previewWidth ?? undefined,
      preview_height: image.previewHeight ?? undefined,
    };
  }
}
