import { OhioAppContext } from '@server/shared/bunker/bunker.interface';
import { saftyParseFloat } from '@shared/helpers';

import {
  OperationType,
  Order,
  OrderItem,
  OrderRefund,
  PaymentType,
  RawOrder,
} from '../ohio.interface';
import { orderItemReducer } from './orderItem';

const DEFAULT_SERVICE = {
  alias: 'unknown',
  id: 'unknown',
  ids: new Set([]),
  name: 'Unknown',
  i18nKeyName: 'unknown',
  iconUrl: '',
};

function isPlusPaymentMethod(paymentMethod: string) {
  return paymentMethod === 'yandex_account_withdraw' || paymentMethod === 'yandex_account_topup';
}

function getValuesFromMap<K, V>(map: Map<K, V>): V[] {
  const result: V[] = new Array(map.size);
  let i = 0;

  for (const value of map.values()) {
    result[i] = value;
    i++;
  }

  return result;
}

interface IReducerOptions {
  isReceipt?: boolean;
}

export function orderReducer(
  order: RawOrder,
  lookup: (groupId: string) => RawOrder[],
  appContext: OhioAppContext,
  options?: IReducerOptions,
): Order {
  const serviceData = order.service_data;
  const { isReceipt } = options || {};

  // Здесь мы собираем композитную транзакцию
  // Бэкэнд должен гарантировать наличие всех составляющих композитной транзакции
  // если они у него есть.
  const children =
    serviceData.payment_method === 'composite' && order.trust_payment_id
      ? lookup(order.trust_payment_id)
      : [order];

  const paymentTypes = children.map<PaymentType>((child) => ({
    method: child.service_data.payment_method,
    price: saftyParseFloat(child.total),
    account: child.service_data.user_account,
    currency: child.currency,
  }));

  const [plusWithdraw, plusTopup, discountValue] = paymentTypes.reduce(
    (aux, child) => {
      const paymentMethod = child.method;

      switch (paymentMethod) {
        case 'yandex_account_withdraw':
          return [aux[0] + child.price, aux[1], aux[2]];
        case 'yandex_account_topup':
          return [aux[0], aux[1] + child.price, aux[2]];
        default:
          if (appContext.discounts.has(paymentMethod)) {
            return [aux[0], aux[1], aux[2] + child.price];
          }

          return aux;
      }
    },
    [0, 0, 0],
  );

  const cashback =
    order.cashback?.type === 'SUCCESS' ? saftyParseFloat(order.cashback.amount) : undefined;

  if (cashback && cashback > 0) {
    paymentTypes.push({
      method: 'cashback_edadeal',
      price: cashback,
      currency: 'RUB',
    });
  }

  const [items, refundOnly] = children.reduce<[OrderItem[], boolean]>(
    (aux, child) => {
      const paymentMethod = child.service_data.payment_method;
      const isChildRefundOnly = child.items.length === 0;

      // Не учитываем плюсы в деталях транзакции так как
      // позиция в заказе будет дубликатом оплаченной деньгами
      if (isPlusPaymentMethod(paymentMethod) || appContext.discounts.has(paymentMethod)) {
        return [aux[0], aux[1] || isChildRefundOnly];
      }

      aux[0].push(...child.items.map(orderItemReducer));

      return [aux[0], aux[1] || isChildRefundOnly];
    },
    [[], false],
  );

  // Сохраняем всю информацию о рефандах
  const refunds = getValuesFromMap(
    children.reduce((aux, child) => {
      child.refunds.forEach(({ items, total }) => {
        const paymentMethod = child.service_data.payment_method;
        const currency = child.currency;

        if (appContext.discounts.has(paymentMethod)) {
          return;
        }

        const key = `${paymentMethod}_${currency}_${child.service_data.user_account}`;

        if (!aux.has(key)) {
          aux.set(key, {
            paymentMethod,
            total: 0,
            items: [],
            currency,
            account: child.service_data.user_account,
          });
        }

        const refund = aux.get(key);

        if (refund && refund.items) {
          refund.total += saftyParseFloat(total);

          if (isPlusPaymentMethod(paymentMethod)) {
            refund.total = Math.round(refund.total);
          } else {
            // Не учитываем плюсы в деталях транзакции
            refund.items = [...refund.items, ...items.map(orderItemReducer)];
          }
        }
      });

      return aux;
    }, new Map<string, OrderRefund>()),
  );

  const service = appContext.getServiceById(order.subservice_id) || DEFAULT_SERVICE;

  let status: string = order.status;

  order.features?.is_advance && (status = 'advance');
  order.features?.operation_type === OperationType.ReturnArrival && (status = 'refunded');

  return {
    id: String(order.order_id),
    orderId: String(order.order_id),
    created: order.created,
    serviceId: order.subservice_id,
    purchaseToken: order.trust_purchase_token,
    paymentId: order.trust_payment_id,
    status,
    refunds,
    items,
    total: saftyParseFloat(order.total) - plusWithdraw - discountValue,
    plus: plusTopup - plusWithdraw,
    cashback,
    currency: order.currency,
    paymentTypes,
    refundOnly,
    receiptsEnabled: appContext.withReceipts(order.subservice_id),
    service: {
      ...service,
      id: service.alias,
    },
    initiatorUid: order.initiator_uid,
    receiptUrl: order.receipt_url,
    retailerName: order.retailer_name,
    retailerIcon: order.retailer_icon,
    source: order.source,
    isReceipt,
  };
}
