import { Injectable, InternalServerErrorException } from '@nestjs/common';
import { BlackboxApiService, UserInfo } from '@server/features/blackbox-api';
import { BlackboxService } from '@server/shared/blackbox';
import { BunkerService } from '@server/shared/bunker';
import { BunkerOhioService } from '@server/shared/bunker/bunker.interface';
import { HttpService } from '@server/shared/http';
import { LoggerService } from '@server/shared/logger';
import { base64Decode, saftyParseFloat } from '@shared/helpers';

import { ordersReducer } from './helpers/ordersReducer';
import {
  ContextOptions,
  EdadealCashbackResponse,
  EdadealDataContext,
  EdadealDataResponse,
  FamilyPayResponse,
  Order,
  OrderList,
  PaymentsDataResponse,
  Receipt,
  ReceiptResponse,
  ServicesResponse,
} from './ohio.interface';

const PLUS = 'plus';
const RECEIPTS = 'receipts';
const LIMIT = 20;
const CURSOR_JOIN = '__';

const parseCursor = (cursor?: string) => {
  if (!cursor) {
    return;
  }

  const parts = base64Decode(cursor).split(CURSOR_JOIN);

  return {
    orderIdKeyset: parts[0],
    createdKeyset: parts[1],
  };
};

const getAllServices = (services: Record<string, BunkerOhioService>) => {
  return Object.values(services).reduce<string[]>((aux, { ids }) => {
    aux.push(...ids);

    return aux;
  }, []);
};

@Injectable()
export class OhioService {
  constructor(
    private http: HttpService,
    private blackboxService: BlackboxService,
    private bunkerService: BunkerService,
    private blackboxApiService: BlackboxApiService,
    private logger: LoggerService,
  ) {}

  async getContext(options: ContextOptions) {
    try {
      const context = await this.bunkerService.getOhioContext();

      return {
        services: await this.getServices(options),
        plusServiceData: context.plusServiceData,
        familyData: await this.getPaymentFamilyData(),
      };
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  /**
   * Список пользователей, которые платили семейной картой текущего пользователя
   */
  async getFamilyPayUsers(): Promise<string[]> {
    try {
      const blackbox = await this.blackboxService.getBlackbox();
      const response = await this.http.get<FamilyPayResponse>(
        `/v1/customer/${blackbox?.uid}/familypay_users`,
      );

      return response.data.data.familypay_users.map(({ initiator_uid }) => String(initiator_uid));
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  async getPaymentFamilyData() {
    try {
      const blackbox = await this.blackboxService.getBlackbox();
      const isFamilyAdmin = blackbox.raw.family_info?.admin_uid === blackbox.uid;

      if (!isFamilyAdmin) {
        return undefined;
      }

      const familyData = await this.blackboxApiService.getFamilyData();

      if (!familyData) {
        return undefined;
      }

      const payers = await this.getFamilyPayUsers();

      // Находим пересечение актуального состава семьи и uid-ов которые платили семейной картой
      // Чтобы в листинге не было платежей бывших участников семьи
      return payers.reduce<UserInfo[]>((acc, item) => {
        const user = familyData.find(({ uid }) => uid === item);

        if (user) {
          acc.push(user);
        }

        return acc;
      }, []);
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  /**
   * Список сервисов на которых пользователь оплачивал что-то
   */
  async getServices(options: ContextOptions) {
    try {
      const blackbox = await this.blackboxService.getBlackbox();
      const response = await this.http.get<ServicesResponse>(
        `/v1/customer/${blackbox?.uid}/services`,
      );
      const { servicesOrder, services } = await this.bunkerService.getOhioContext();

      return servicesOrder.reduce<any[]>((aux, alias) => {
        const service = services[alias];

        const exists = response.data.data.services.some((backendService) => {
          return service.ids.has(backendService.subservice_id);
        });

        if (alias === PLUS && options.familyMode) {
          return aux;
        }

        if (exists || alias === PLUS) {
          aux.push({
            id: service.alias,
            ...service,
          });
        }

        return aux;
      }, []);
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  /**
   * Список платежей
   */
  async getPayments(params: any): Promise<OrderList> {
    try {
      const { alias, user, after } = params;
      const { orderIdKeyset, createdKeyset } = parseCursor(after) || {};

      const blackbox = await this.blackboxService.getBlackbox();
      const context = await this.bunkerService.getOhioContext();
      const { services } = context;

      if (alias === PLUS) {
        return await this.getPlusPayments(orderIdKeyset, createdKeyset);
      }

      if (alias === RECEIPTS) {
        return await this.getEdadealPayments(orderIdKeyset, createdKeyset);
      }

      const response = await this.http.get<PaymentsDataResponse>(
        `/v1/customer/${blackbox?.uid}/orders`,
        {
          params: {
            subservice_ids:
              alias && services[alias] ? Array.from(services[alias].ids) : getAllServices(services),
            limit: LIMIT,
            order_id_keyset: orderIdKeyset,
            created_keyset: createdKeyset,
            initiator_uid: user !== blackbox.uid ? user : undefined,
            'show-yandexpay': true,
            'hide-family-payments': user === blackbox.uid,
          },
        },
      );

      return ordersReducer(response.data.data, context, this.logger);
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  async getPaymentById(paymentId: string): Promise<Order[]> {
    try {
      const blackbox = await this.blackboxService.getBlackbox();
      const context = await this.bunkerService.getOhioContext();

      const response = await this.http.get<PaymentsDataResponse>(
        `/v1/customer/${blackbox?.uid}/payment`,
        {
          params: {
            trust_payment_id: paymentId,
          },
        },
      );

      return ordersReducer(response.data.data, context, this.logger).edges;
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  /**
   * Список "плюсовых" платежей
   */
  async getPlusPayments(
    orderIdKeyset: string | undefined,
    createdKeyset: string | undefined,
  ): Promise<OrderList> {
    try {
      const blackbox = await this.blackboxService.getBlackbox();
      const context = await this.bunkerService.getOhioContext();

      const response = await this.http.get<PaymentsDataResponse>(
        `/v1/customer/${blackbox?.uid}/yandex_account_orders`,
        {
          params: {
            limit: LIMIT,
            order_id_keyset: orderIdKeyset,
            created_keyset: createdKeyset,
          },
        },
      );

      return ordersReducer(response.data.data, context, this.logger);
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  /**
   * Список электрочеков (Едадил)
   */
  async getEdadealPayments(
    orderIdKeyset: string | undefined,
    createdKeyset: string | undefined,
  ): Promise<OrderList> {
    try {
      const blackbox = await this.blackboxService.getBlackbox();
      const context = await this.bunkerService.getOhioContext();

      const response = await this.http.get<PaymentsDataResponse>(
        `/v1/customer/${blackbox?.uid}/fns_orders`,
        {
          params: {
            limit: LIMIT,
            order_id_keyset: orderIdKeyset,
            created_keyset: createdKeyset,
          },
        },
      );

      return ordersReducer(response.data.data, context, this.logger, { isReceipt: true });
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  /**
   * Баланс кэшбека в Едадиле
   */
  async getCashbackBalance(): Promise<number> {
    try {
      const blackbox = await this.blackboxService.getBlackbox();
      const response = await this.http.get<EdadealCashbackResponse>(
        `/v1/customer/${blackbox?.uid}/cashback_balance`,
      );

      return Math.floor(saftyParseFloat(response.data.data.amount));
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  /**
   * Данные привязки к Едадилу
   */
  async getEdadealData(): Promise<EdadealDataContext> {
    try {
      const blackbox = await this.blackboxService.getBlackbox();
      const response = await this.http.get<EdadealDataResponse>(
        `/v1/customer/${blackbox?.uid}/fns_binding_status`,
      );
      const { status, urls } = response.data;
      const hasBinding = status === 'has_binding';

      return {
        hasBinding,
        amount: hasBinding ? await this.getCashbackBalance() : null,
        url: urls.touch,
        iframeSrc: urls.desktop,
      };
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }

  async getReceiptByPurchaseToken(purchaseToken: string): Promise<Receipt[]> {
    try {
      const response = await this.http.get<ReceiptResponse>('/receipts', {
        params: { purchase_token: purchaseToken },
      });

      return response.data.data.receipts.map(({ url, payments, timestamp }) => {
        const amount = payments.map(({ amount }) => saftyParseFloat(amount));
        const type = payments?.[0]?.payment_type;

        return {
          url,
          amount,
          timestamp,
          type,
        };
      });
    } catch (error) {
      throw new InternalServerErrorException(error);
    }
  }
}
