import { Injectable } from '@nestjs/common';
import {
  ProfileFieldValidationResultCode,
  UpdatePersonalInfoInput,
  UpdatePersonalInfoPayload,
  UpdatePersonalInfoProblem,
  UpdatePersonalInfoProblemKind,
} from '@server/graphql-schema';
import { BlackboxService } from '@server/shared/blackbox';
import { BLACKBOX_ALIASES, __TODO_BLACKBOX_ATTRIBUTES } from '@server/shared/blackbox/constants';
import { LangDetectService } from '@server/shared/lang-detect';
import { fromModel } from '@server/shared/libs';
import { PassportService } from '@server/shared/passport';
import { ValidationFieldName } from '@server/shared/passport/api-types';
import { IpregService } from '@yandex-id/ipreg';
import { Context } from '@yandex-int/nest-common';

import { LocationService } from '../location/location.service';

@Injectable()
export class ViewerService {
  // eslint-disable-next-line max-params
  constructor(
    private context: Context,
    private blackboxService: BlackboxService,
    private langDetectService: LangDetectService,
    private ipregService: IpregService,
    private locationService: LocationService,
    private passportService: PassportService,
  ) {}

  async getCommonUserData(shouldForceCache = false) {
    const [blackbox, location] = await Promise.all([
      this.blackboxService.getBlackbox(shouldForceCache),
      this.locationService.getLocation(),
    ]);

    const isYandexoid = blackbox.raw.aliases[BLACKBOX_ALIASES.yandexoid];
    const isYandexNet = this.ipregService.isUserNet();

    return {
      isYandexoid,
      isYandexNet,
      avatarId: blackbox.avatar?.default ?? '0/0-0',
      havePlus: blackbox.havePlus === '1',
      language: this.langDetectService.userLang ?? 'ru',
      uid: blackbox.uid,
      displayName: blackbox.raw.display_name.public_name,
      fullName: blackbox.fio,
      isChild: blackbox.isChild === '1',
      city: blackbox.city,
      isFamilyAdmin: blackbox.raw.family_info?.admin_uid === blackbox.uid,
      firstName: blackbox.firstName,
      lastName: blackbox.lastName,
      birthdate: blackbox.birthdate,
      publicId: blackbox.raw.public_id,
      hasPublicIdSet: blackbox.hasPublicIdSet === '1',
      gender: blackbox.gender ?? 'UNKNOWN',
      country: blackbox.country ? blackbox.country : location.countryCode,
      timezone: blackbox.timezone ? blackbox.timezone : location.timezone,
      hasFamilyPlus: blackbox.hasFamilyPlus === '1',
    };
  }

  async updateUserData(input: UpdatePersonalInfoInput) {
    const personalInfoResult = await this.passportService.updateProfile(input);

    if (input.deleteAvatar) {
      const deleteAvatarResponse = await this.passportService.deleteDefaultAvatar();

      if (deleteAvatarResponse.status === 'error') {
        return fromModel(UpdatePersonalInfoProblem, {
          reason: UpdatePersonalInfoProblemKind.DELETE_AVATAR_INTERNAL,
        });
      }
    }

    if (personalInfoResult.status === 'error') {
      return fromModel(UpdatePersonalInfoProblem, {
        reason: UpdatePersonalInfoProblemKind.UPDATE_DATA_INTERNAL,
      });
    }

    return fromModel(UpdatePersonalInfoPayload, {});
  }

  async validatePersonalInfoField(fieldName: ValidationFieldName, fieldValue: string) {
    const result = await this.passportService.validateProfileField(fieldName, fieldValue);
    let code = ProfileFieldValidationResultCode.OK;

    if (result.status === 'error') {
      if (result.errors.includes(`${fieldName}.invalid`)) {
        code = ProfileFieldValidationResultCode.INVALID;
      } else {
        code = ProfileFieldValidationResultCode.INTERNAL;
      }
    }

    return { code };
  }

  deleteAvatar() {
    return this.passportService.deleteDefaultAvatar();
  }
}
