import { BunkerData, bunker } from 'luster';

import { Injectable, InternalServerErrorException } from '@nestjs/common';
import { BunkerConfigService } from '@server/config';
import { LoggerService } from '@server/shared/logger';
import Bunker from '@vertis/luster-bunker/lib/bunker';

import { OhioAppContext, RawOhioBunker } from './bunker.interface';

function parseUrl(url: string | undefined, getData: NodeGetter): string | undefined {
  if (url?.startsWith('bunker://')) {
    // Пока плагин возвращает плоский список узлов с полными путями через "/"
    // return get(config, url.slice(9).replace(/\//g, '.'), '');
    return getData(url.slice(9)) || '';
  }

  return url;
}

type NodeGetter = <TResult>(...paths: string[]) => TResult;

@Injectable()
export class BunkerService {
  private nodesData: BunkerData<RawOhioBunker>;
  constructor(private bunkerConfig: BunkerConfigService, private logger: LoggerService) {
    // В режиме кластера получаем данных от мастер-процесса
    if (bunker) {
      this.nodesData = bunker.configureAPI<RawOhioBunker>();
      // иначе это режим разработки и в нем реализуем простейший поход за данными без обновления
    } else {
      const { timeout, apiHost, ohioProject, version } = this.bunkerConfig.options;
      const bunker = new Bunker({
        family: 6,
        timeout: timeout,
        maxRetries: 10,
        requestId: 'luster-bunker',
        host: apiHost,
      });
      const nodes = {
        node: ohioProject,
        version: version,
      };
      let nodesData: RawOhioBunker;

      bunker.loadNodes<RawOhioBunker>([nodes]).then((data) => (nodesData = data));

      this.getBunkerData = <TResult>(project?: string, ...paths: string[]) => {
        const path = this.getPathString(project, ...paths);

        if (!path) {
          return nodesData;
        }

        // тут типы свести практически невозможно и с учетом возможного изменения данных в бункере не особо что-то гарантирует
        return this.parseAvatarUrl(
          path,
          nodesData[path as keyof RawOhioBunker] as unknown as TResult,
        );
      };
    }
  }

  getBunkerData(): RawOhioBunker;
  getBunkerData<TResult>(project?: string, ...paths: string[]): TResult;
  getBunkerData<TResult>(project?: string, ...paths: string[]) {
    const path = this.getPathString(project, ...paths);
    const data = this.nodesData.getNode<TResult>(path);

    if (path) {
      return this.parseAvatarUrl(path, data);
    }

    return data;
  }

  // Ужасный костыль. Исправляться будет тут https://st.yandex-team.ru/PASSP-37904
  parseAvatarUrl<TResult>(path: string, data: TResult | string): TResult | string {
    if (path.indexOf('/images/') > -1 && typeof data === 'string') {
      // из строки вида ...<image xlink:href="https://avatars.mds.yandex.net/get-bunker/128809/0bfa0f5257932853fc91891556c333ee4258b773/svg" width="10...
      // получаем только ссылку

      const match = /xlink:href="([^"]+)"/.exec(data);

      if (match && match[1]) {
        return match[1];
      }
    }

    return data;
  }

  projectGetter(project: string): NodeGetter {
    return <TResult>(...paths: string[]) => this.getBunkerData<TResult>(project, ...paths);
  }

  getOhioContext(): OhioAppContext {
    try {
      const { ohioProject } = this.bunkerConfig.options;
      const getData = this.projectGetter(ohioProject);

      const config = getData<RawOhioBunker['config']>('config');
      const services = Object.keys(config.services).reduce<OhioAppContext['services']>(
        (aux, key) => {
          const item = config.services[key];

          if (!item) {
            return aux;
          }

          aux[key] = {
            alias: key,
            iconUrl: parseUrl(item.iconUrl, getData),
            splashUrl: parseUrl(item.splashUrl, getData),
            ids: new Set(item.ids),
            name: item.name,
            i18nKeyName: item.i18nKeyName,
            url: parseUrl(item.url, getData),
            help: parseUrl(item.help, getData),
          };

          return aux;
        },
        {},
      );

      return {
        services,
        servicesOrder: config.order,
        discounts: new Set(config.discounts),
        payServicesPlusPromo: config.payServicesPlusPromo,
        plusServiceData: {
          ...config.plusServiceData,
          splashUrl: parseUrl(config.plusServiceData.splashUrl, getData) || '',
          iconUrl: parseUrl(config.plusServiceData.iconUrl, getData) || '',
        },
        withReceipts: (id: string) => !config.servicesIdsWithoutCheck.includes(id),
        getServiceById: (id: string) =>
          Object.values(services).find((service) => {
            return service.ids.has(id);
          }),
      };
    } catch (error) {
      this.logger.error('Get ohio context error', error);

      throw new InternalServerErrorException(error);
    }
  }

  private getPathString(project?: string, ...paths: string[]) {
    return project ? project + `/${paths.join('/')}` : undefined;
  }
}
