type EnumToString<T> = T extends `${infer P}` ? P : never;

type HttpDelineators = '/' | '?' | '#' | '\\';

type OptionalPath = `${HttpDelineators}${string}` | '';

type HostProtocolSchemes = `${string}://` | '';

type PortScheme = `:${number}` | '' | ':*';

type HostNameScheme = `${string}.${string}` | `localhost`;

export type ValidCrypto = 'nonce' | 'sha256' | 'sha384' | 'sha512';

export enum Source {
  /* schema sources */
  HTTP = 'http:',
  HTTPS = 'https:',
  DATA = 'data:',
  MEDIASTREAM = 'mediastream:',
  BLOB = 'blob:',
  FILESYSTEM = 'filesystem:',

  /* base sources */
  SELF = 'self',
  UNSAFE_EVAL = 'unsafe-eval',
  UNSAFE_HASHES = 'unsafe-hashes',
  UNSAFE_INLINE = 'unsafe-inline',
  NONE = 'none',

  /* action sources */
  STRICT_DYNAMIC = 'strict-dynamic',
  REPORT_SAMPLE = 'report-sample',

  /* crypto */
  NONCE = '%nonce%',
}

export type SchemeSources =
  | Source.HTTP
  | Source.HTTPS
  | Source.DATA
  | Source.MEDIASTREAM
  | Source.BLOB
  | Source.FILESYSTEM;

export type BaseSources =
  | Source.SELF
  | Source.UNSAFE_EVAL
  | Source.UNSAFE_HASHES
  | Source.UNSAFE_INLINE
  | Source.NONE;

export type UriPath = `${HttpDelineators}${string}`;

export type UrlString = `${HostSource}${OptionalPath}`;

export type HostSource = `${HostProtocolSchemes}${HostNameScheme}${PortScheme}`;

export type CryptoSources = `${ValidCrypto}-${string}` | Source.NONCE;

export type Sources = EnumToString<BaseSources | SchemeSources | HostSource | CryptoSources>;

export type ActionSources = EnumToString<Sources | Source.STRICT_DYNAMIC | Source.REPORT_SAMPLE>;

export type FrameSources = EnumToString<HostSource | SchemeSources | Source.SELF | Source.NONE>;

export type PluginSources = EnumToString<`${string}/${string}` | Source.NONE>;

export enum SandboxOption {
  ALLOW_DOWNLOADS_WITHOUT_USER_ACTIVATION = 'allow-downloads-without-user-activation',
  ALLOW_FORMS = 'allow-forms',
  ALLOW_MODALS = 'allow-modals',
  ALLOW_ORIENTATION_LOCK = 'allow-orientation-lock',
  ALLOW_POINTER_LOCK = 'allow-pointer-lock',
  ALLOW_POPUPS = 'allow-popups',
  ALLOW_POPUPS_TO_ESCAPE_SANDBOX = 'allow-popups-to-escape-sandbox',
  ALLOW_PRESENTATION = 'allow-presentation',
  ALLOW_SAME_ORIGIN = 'allow-same-origin',
  ALLOW_SCRIPTS = 'allow-scripts',
  ALLOW_STORAGE_ACCESS_BY_USER_ACTIVATION = 'allow-storage-access-by-user-activation',
  ALLOW_TOP_NAVIGATION = 'allow-top-navigation',
  ALLOW_TOP_NAVIGATION_BY_USER_ACTIVATION = 'allow-top-navigation-by-user-activation',
}

export enum ReferrerHeaderOption {
  NO_REFERRER = 'no-referrer',
  NO_REFERRER_WHEN_DOWNGRADE = 'no-referrer-when-downgrade',
  ORIGIN = 'origin',
  ORIGIN_WHEN_CROSS_ORIGIN = 'origin-when-cross-origin',
  SAME_ORIGIN = 'same-origin',
  STRICT_ORIGIN = 'strict-origin',
  STRICT_ORIGIN_WHEN_CROSS_ORIGIN = 'strict-origin-when-cross-origin',
  UNSAFE_URL = 'unsafe-url',
  NONE = 'none',
}

export enum SriPolicy {
  SCRIPT = 'script',
  STYLE = 'style',
  SCRIPT_STYLE = 'script style',
}

export enum RequireTrustedTypePolicy {
  SCRIPT = 'script',
}

export enum TrustedTypesPolicy {
  NONE = 'none',
  ALLOW_DUPLICATES = 'allow-duplicates',
  ANY = '*',
}

export type TrustedTypesPolicies = TrustedTypesPolicy | string;

export interface ReportToEndpoint {
  url: UrlString;
}

export interface ReportTo {
  group: string;
  max_age: number;
  endpoints: ReportToEndpoint[];
  include_subdomains?: boolean;
}

export enum Directive {
  /* child directives */
  CHILD_SRC = 'child-src',
  WORKER_SRC = 'worker-src',
  FRAME_SRC = 'frame-src',

  /* source directives */
  CONNECT_SRC = 'connect-src',
  DEFAULT_SRC = 'default-src',
  FONT_SRC = 'font-src',
  IMG_SRC = 'img-src',
  MANIFEST_SRC = 'manifest-src',
  MEDIA_SRC = 'media-src',
  OBJECT_SRC = 'object-src',
  PREFETCH_SRC = 'prefetch-src',
  SCRIPT_SRC = 'script-src',
  SCRIPT_SRC_ELEM = 'script-src-elem',
  SCRIPT_SRC_ATTR = 'script-src-attr',
  STYLE_SRC = 'style-src',
  STYLE_SRC_ELEM = 'style-src-elem',
  STYLE_SRC_ATTR = 'style-src-attr',

  /* document directives */
  BASE_URI = 'base-uri',
  /**
   * @deprecated
   */
  PLUGIN_TYPES = 'plugin-types',
  SANDBOX = 'sandbox',

  /* navigation direactives */
  FORM_ACTION = 'form-action',
  FRAME_ANCESTORS = 'frame-ancestors',
  NAVIGATE_TO = 'navigate-to',

  /* reporting directives */
  /**
   * @deprecated
   */
  REPORT_URI = 'report-uri',
  REPORT_TO = 'report-to',

  /* other directives */
  /**
   * @deprecated
   */
  BLOCK_ALL_MIXED_CONTENT = 'block-all-mixed-content',
  /**
   * @deprecated
   */
  REFERRER = 'referrer',
  /**
   * @deprecated
   */
  REQUIRE_SRI_FOR = 'require-sri-for',
  REQUIRE_TRUSTED_TYPES_FOR = 'require-trusted-types-for',
  TRUSTED_TYPES = 'trusted-types',
  UPGRADE_INSECURE_REQUESTS = 'upgrade-insecure-requests',
}

export interface ChildDirectives {
  [Directive.CHILD_SRC]?: Sources[];
  [Directive.FRAME_SRC]?: Sources[];
  [Directive.WORKER_SRC]?: Sources[];
}

export interface SourceDirectives {
  [Directive.CONNECT_SRC]?: Sources[];
  [Directive.DEFAULT_SRC]?: ActionSources[];
  [Directive.FONT_SRC]?: Sources[];
  [Directive.IMG_SRC]?: Sources[];
  [Directive.MANIFEST_SRC]?: Sources[];
  [Directive.MEDIA_SRC]?: Sources[];
  [Directive.OBJECT_SRC]?: Sources[];
  [Directive.PREFETCH_SRC]?: Sources[];
  [Directive.SCRIPT_SRC]?: ActionSources[];
  [Directive.SCRIPT_SRC_ELEM]?: Sources[];
  [Directive.SCRIPT_SRC_ATTR]?: Sources[];
  [Directive.STYLE_SRC]?: Sources[];
  [Directive.STYLE_SRC_ELEM]?: Sources[];
  [Directive.STYLE_SRC_ATTR]?: Sources[];
}

export interface DocumentDirectives {
  [Directive.BASE_URI]?: ActionSources[];
  /**
   * @deprecated
   */
  [Directive.PLUGIN_TYPES]?: PluginSources[];
  [Directive.SANDBOX]?: SandboxOption;
}

export interface NavigationDirectives {
  [Directive.FORM_ACTION]?: ActionSources[];
  [Directive.FRAME_ANCESTORS]?: FrameSources[];
  [Directive.NAVIGATE_TO]?: ActionSources[];
}

export interface ReportingDirectives {
  /**
   * @deprecated
   */
  [Directive.REPORT_URI]?: UriPath;
  [Directive.REPORT_TO]?: ReportTo['group'];
}

export interface OtherDirectives {
  /**
   * @deprecated
   */
  [Directive.BLOCK_ALL_MIXED_CONTENT]?: boolean;
  /**
   * @deprecated
   */
  [Directive.REFERRER]?: ReferrerHeaderOption;
  /**
   * @deprecated
   */
  [Directive.REQUIRE_SRI_FOR]?: SriPolicy;
  [Directive.REQUIRE_TRUSTED_TYPES_FOR]?: RequireTrustedTypePolicy;
  [Directive.TRUSTED_TYPES]?: TrustedTypesPolicies[];
  [Directive.UPGRADE_INSECURE_REQUESTS]?: boolean;
}

export type Directives = ChildDirectives &
  SourceDirectives &
  OtherDirectives &
  ReportingDirectives &
  NavigationDirectives &
  DocumentDirectives;

export type DirectiveNames = keyof Directives;

export type DirectiveValues = Directives[DirectiveNames];
