import { DynamicModule, MiddlewareConsumer, Module, NestModule, Provider } from '@nestjs/common';
import { CookieModule } from '@yandex-int/nest-common';

import { CSP_DEFAULT_OPTIONS, CSP_OPTIONS_TOKEN } from './csp-internal.constants';
import { CSPModuleAsyncOptions, CSPModuleOptions, CSPOptionsFactory } from './csp.interfaces';
import { CSPMiddleware } from './csp.middleware';
import { CSPService } from './csp.service';

@Module({
  imports: [CookieModule],
  providers: [
    {
      provide: CSP_OPTIONS_TOKEN,
      useValue: CSP_DEFAULT_OPTIONS,
    },
    CSPService,
  ],
  exports: [CSPService],
})
export class CSPModule implements NestModule {
  static forRoot(options: CSPModuleOptions): DynamicModule {
    const { isGlobal, ...cspOptions } = options;

    return {
      module: CSPModule,
      global: isGlobal,
      providers: [{ provide: CSP_OPTIONS_TOKEN, useValue: cspOptions }],
    };
  }

  static forRootAsync(options: CSPModuleAsyncOptions): DynamicModule {
    return {
      module: CSPModule,
      global: options.isGlobal,
      imports: options.imports,
      providers: this.createAsyncProviders(options),
    };
  }

  private static createAsyncProviders(options: CSPModuleAsyncOptions): Provider[] {
    const providers: Provider[] = [this.createAsyncOptionsProvider(options)];

    if (!options.useExisting && options.useClass) {
      providers.push({ provide: options.useClass, useClass: options.useClass });
    }

    return providers;
  }

  private static createAsyncOptionsProvider(options: CSPModuleAsyncOptions): Provider {
    const { useFactory, useExisting, useClass, inject = [] } = options;

    if (useFactory) {
      return {
        provide: CSP_OPTIONS_TOKEN,
        useFactory,
        inject,
      };
    }

    const dep = useExisting || useClass;

    if (dep) {
      return {
        provide: CSP_OPTIONS_TOKEN,
        async useFactory(optionsFactory: CSPOptionsFactory) {
          return await optionsFactory.createCSPOptions();
        },
        inject: [dep],
      };
    }

    throw new Error('Invalid configuration. Must provide useFactory, useClass or useExisting');
  }

  configure(consumer: MiddlewareConsumer) {
    consumer
      .apply(CSPMiddleware)
      .exclude('/_next/(.*)', '/pay/payment/frame-3ds', '/pay/payment/3ds-complete')
      .forRoutes('*');
  }
}
