import crypto from 'crypto';

import { CSPDirectives, CSPPresetsObject, getCSP, nonce as wrapNonce } from 'csp-header';
import * as psl from 'psl';

import { Inject, Injectable } from '@nestjs/common';
import { Context, CookieService, StorageMemoize } from '@yandex-int/nest-common';

import {
  Directive,
  DirectiveNames,
  DirectiveValues,
  Directives,
  ReportTo,
  Source,
  UrlString,
} from './csp-core-types';
import {
  CSP_OPTIONS_TOKEN,
  DEFAULT_DOMAIN_OPTIONS,
  DEFAULT_REPORT_GROUP,
  HEADERS,
  SINGLE_QUOTED_SOURCES,
  VALID_CRYPTO,
} from './csp-internal.constants';
import { AUTO_TLD } from './csp.constants';
import {
  CSPOptions,
  CSPPresets,
  DefaultReportUriParams,
  DomainOptions,
  NormalizedCSPOptions,
} from './csp.interfaces';

interface CSPHeader {
  name: string;
  value: string;
}

type UnwrapArray<T> = T extends Array<infer P> ? P : T;

type DirectiveValue = UnwrapArray<DirectiveValues>;

@Injectable()
export class CSPService {
  constructor(
    @Inject(CSP_OPTIONS_TOKEN) private rawOptions: CSPOptions,
    private context: Context,
    private cookieService: CookieService,
  ) {}

  get options() {
    return this.normalizeOptions(this.rawOptions);
  }

  @StorageMemoize()
  getNonce() {
    const nonce = crypto.randomBytes(16).toString('base64');

    return nonce;
  }

  @StorageMemoize()
  getHeaders() {
    const headers: CSPHeader[] = [];
    const cspHeader = this.getCspHeader();

    headers.push(cspHeader);

    const reportToHeader = this.getReportToHeader();

    if (reportToHeader) {
      headers.push(reportToHeader);
    }

    return headers;
  }

  @StorageMemoize()
  getCspHeader(): CSPHeader {
    const { reportOnly, reportUri, directives, presets } = this.options;
    const name = reportOnly ? HEADERS.CSP_REPORT_ONLY : HEADERS.CSP;
    const rawValue = getCSP({ directives, presets, reportUri });
    const value = this.interpolateCspHeader(rawValue);

    return {
      name,
      value,
    };
  }

  @StorageMemoize()
  getReportToHeader(): CSPHeader | null {
    const { reportTo } = this.options;

    if (!reportTo) {
      return null;
    }

    return {
      name: HEADERS.REPORT_TO,
      value: reportTo.map((group) => JSON.stringify(group)).join(','),
    };
  }

  private normalizeOptions(options: CSPOptions) {
    const {
      enableDefaultReportUri = false,
      enableDefaultReportTo = true,
      defaultReportUriParams = {},
      reportTo,
      reportUri,
      directives,
      presets,
      domainOptions = DEFAULT_DOMAIN_OPTIONS,
      reportOnly,
    } = options;

    const params: NormalizedCSPOptions = {
      domainOptions,
      reportOnly,
    };

    if (enableDefaultReportUri) {
      const reportUri = this.getDefaultReportUri(defaultReportUriParams);

      params.reportUri = reportUri;

      if (enableDefaultReportTo) {
        params.directives = {
          ...params.directives,
          [Directive.REPORT_TO]: DEFAULT_REPORT_GROUP,
        };

        params.reportTo = this.getDefaultReportTo(reportUri);
      }
    } else {
      const { req, res } = this.context;

      params.reportUri = typeof reportUri === 'function' ? reportUri(req, res) : reportUri;
      params.reportTo = typeof reportTo === 'function' ? reportTo(req, res) : reportTo;
    }

    if (directives) {
      params.directives = this.normalizeDirectives(directives);
    }

    if (presets) {
      params.presets = this.normalizePresets(presets);
    }

    return params;
  }

  private normalizePresets(presets: CSPPresets) {
    if (Array.isArray(presets)) {
      return presets.map((preset) => {
        return this.normalizeDirectives(preset);
      });
    }

    const presetMap: CSPPresetsObject = {};

    for (const presetName of Object.keys(presets)) {
      const preset = presets[presetName];

      if (preset) {
        presetMap[presetName] = this.normalizeDirectives(preset);
      }
    }

    return presetMap;
  }

  private normalizeDirectives(directives: Directives): Partial<CSPDirectives> {
    const result: Partial<CSPDirectives> = {};

    for (const directiveName of Object.keys(directives) as DirectiveNames[]) {
      const rawValue = directives[directiveName];
      const value = this.normalizeSourceValue(rawValue);

      Object.assign(result, {
        [directiveName]: value,
      });
    }

    return result;
  }

  private normalizeSourceValue(value: DirectiveValues) {
    if (Array.isArray(value)) {
      return value.map((v) => this.ensureSourceValue(v));
    }

    return this.ensureSourceValue(value);
  }

  private ensureSourceValue<T extends DirectiveValue>(value: T): T {
    if (typeof value === 'string') {
      if (SINGLE_QUOTED_SOURCES.has(value) || VALID_CRYPTO.some((v) => value.startsWith(v))) {
        return `'${value}'` as T;
      }
    }

    return value;
  }

  private interpolateCspHeader(value: string) {
    const replacers = [
      (value: string) => this.applyNonce(value),
      (value: string) => this.applyAutoTld(value),
    ];

    return replacers.reduce((acc, fn) => fn(acc), value);
  }

  private applyNonce(value: string) {
    if (!value.includes(Source.NONCE)) {
      return value;
    }

    const nonce = this.getNonce();

    return value.replace(new RegExp(Source.NONCE, 'g'), wrapNonce(nonce));
  }

  private applyAutoTld(value: string) {
    if (!value.includes(AUTO_TLD)) {
      return value;
    }

    const { domainOptions } = this.options;
    const { hostname } = this.context.req;

    const tld = this.parseDomain(hostname, domainOptions);

    if (!tld) {
      return value;
    }

    return value.replace(new RegExp(AUTO_TLD, 'g'), tld);
  }

  private parseDomain(hostname: string, options: DomainOptions = {}): string | null {
    const { customTlds } = options;

    if (customTlds instanceof RegExp) {
      const tld = hostname.match(customTlds);

      if (tld !== null) {
        return tld[0].replace(/^\.+/, '');
      }
    }

    if (Array.isArray(customTlds)) {
      for (const tld of customTlds) {
        if (hostname.endsWith(`.${tld}`)) {
          return tld;
        }
      }
    }

    const domain = psl.parse(hostname);

    if (domain.error) {
      return null;
    }

    return domain.tld;
  }

  private getDefaultReportUri(options: DefaultReportUriParams = {}): UrlString {
    const { from, project } = options;
    const { yandex_login: login, yandexuid: uid } = this.cookieService.getCookies();
    const searchParams = new URLSearchParams();

    if (login) {
      searchParams.set('yandex_login', login);
    }

    if (uid) {
      searchParams.set('yandexuid', uid);
    }

    if (from) {
      searchParams.set('from', from);
    }

    if (project) {
      searchParams.set('project', project);
    }

    return `https://csp.yandex.net/csp?${searchParams.toString()}`;
  }

  private getDefaultReportTo(url: UrlString): ReportTo[] {
    return [
      {
        group: DEFAULT_REPORT_GROUP,
        endpoints: [{ url }],
        max_age: 30 * 60,
        include_subdomains: true,
      },
    ];
  }
}
