import { Request } from 'express';
import { RenderFilter, RenderService } from 'nest-next';
import { Observable, throwError } from 'rxjs';
import { catchError } from 'rxjs/operators';

import {
  ArgumentsHost,
  CallHandler,
  Catch,
  ExecutionContext,
  Injectable,
  NestInterceptor,
  NotFoundException,
} from '@nestjs/common';
import { HttpAdapterHost } from '@nestjs/core';
import { LoggerService } from '@server/shared/logger';

@Catch()
@Injectable()
export class ErrorsLogInterceptor extends RenderFilter implements NestInterceptor {
  constructor(
    protected readonly httpAdapterHost: HttpAdapterHost,
    private renderService: RenderService,
    private logger: LoggerService,
  ) {
    super(renderService);
  }

  async catch(err: unknown, host: ArgumentsHost) {
    if (this.shouldLog(err)) {
      this.logError(err, host.switchToHttp().getRequest());
    }

    return super.catch(err, host);
  }

  intercept(context: ExecutionContext, next: CallHandler): Observable<any> {
    return next.handle().pipe(
      catchError((err: any) => {
        this.logError(err);

        return throwError(() => err);
      }),
    );
  }

  shouldLog(err: any) {
    // запросы за статикой обслуживаются сервером некса, поэтому тут они проявляются как ошибка — их игнорим
    return !(
      typeof err === 'object' &&
      err instanceof NotFoundException &&
      (err.message.indexOf('Cannot GET /_next/static/') === 0 ||
        err.message.indexOf('Cannot GET /_next/image') === 0)
    );
  }

  logError(err: any, req?: Request) {
    let msg = '';

    if (req?.originalUrl) {
      msg += `Handling: ${req?.originalUrl}\n`;
    }
    if (err.request?._currentUrl) {
      msg += `Requested url: ${err.request?._currentUrl}\n`;
    }
    if (err.code) {
      msg += `Code: ${err.code}\n`;
    }

    if (err.response?.data) {
      msg += `Response: ${JSON.stringify(err.response.data, null, 4)}\n`;
    }

    this.logger.error(msg, err.stack);
  }
}
