import { AxiosResponseHeaders } from 'axios';

import { Injectable } from '@nestjs/common';
import { ExperimentsConfigService, ExperimentsOptions } from '@server/config';
import { HttpService } from '@server/shared/http';
import { LoggerService } from '@server/shared/logger';
import { base64Decode, base64Encode, safelyParseJSON } from '@shared/helpers';
import { Dict } from '@shared/types/common';
import { Context, CookieService, RequestIdService, StorageMemoize } from '@yandex-int/nest-common';
import { UatraitsService } from '@yandex-int/nest-infra';

import { Experiments, ExperimentsRawData } from './experiments.interface';

@Injectable()
export class ExperimentsService {
  private static HANDLER_NAME = 'PASSPORT' as const;
  private config: ExperimentsOptions;

  // eslint-disable-next-line max-params
  constructor(
    private configService: ExperimentsConfigService,
    private http: HttpService,
    private logger: LoggerService,
    private cookieService: CookieService,
    private uatraitsService: UatraitsService,
    private requestIdService: RequestIdService,
    private context: Context,
  ) {
    this.config = this.configService.options;
  }

  @StorageMemoize()
  public async getExperiments() {
    const [ab, ecoo] = await Promise.all([this.getExperimentsAB(), this.getExperimentsSticky()]);

    ab.boxes.push(...ecoo.boxes);
    Object.assign(ab.flags, ecoo.flags);

    return ab;
  }

  /**
   * Получение экспериментов пользователя из AB
   * Получаются только честные эксперименты из юзерсплиттера, в которые попал пользователь (без залипания)
   * Из полученных экспериментов возвращаются только применимые, валидные, с нужным хендлером
   *
   * @async
   * @returns {Experiments}
   */
  @StorageMemoize()
  async getExperimentsAB(): Promise<Experiments> {
    const experimentsData = await this.getFromUAAS();

    if (!experimentsData?.flags) {
      return this.getEmptyExperiments();
    }

    const exps = this.getApliedFlags(experimentsData);

    this.logger.debug('Loaded AB exps', { exps });

    return exps;
  }

  /**
   * Получение экспериментов из залипания пользователя
   * Получается список экспов пользователя, в которые он залип и возвращаются только с нужным хендлером
   *
   * @async
   * @returns {Experiments}
   */
  @StorageMemoize()
  async getExperimentsSticky(): Promise<Experiments> {
    const experimentsData = await this.getFromECOO();

    if (!experimentsData?.flags) {
      return this.getEmptyExperiments();
    }

    const exps = this.getApliedFlags(experimentsData);

    this.logger.debug('Loaded sticky exps', { exps });

    return exps;
  }

  private getEmptyExperiments(): Experiments {
    return {
      flags: {},
      boxes: [],
    };
  }

  private decodeExp(expData: string): { flags: Object | string[] } | null {
    if (!expData) {
      return null;
    }

    const decodedExpData = base64Decode(expData);
    let exp;

    try {
      exp = JSON.parse(decodedExpData);
    } catch (e) {
      this.logger.warn("Can't parse experiment", { err: e, expData, decodedExpData });
    }

    // Если во флаге нет нужного хендлера, то игнорируем его (флаг) [просто левый эксп]
    const handlerValue = exp?.[0]?.CONTEXT?.[ExperimentsService.HANDLER_NAME];

    if (!handlerValue) {
      return null;
    }

    return handlerValue;
  }

  /**
   * Проверяет применимость тестидов к выдаче
   * Декодирует флаги, фильтрует по правильному хендлеру в данных и невалидные
   *
   * @param {ExperimentsRawData} expData - сырые данные флагов от АБшницы
   * @returns {Object}
   */
  private getApliedFlags(expData: ExperimentsRawData) {
    return expData.flags.reduce<Experiments>(
      (acc, encExp, idx) => {
        const exp = this.decodeExp(encExp);

        // Если флаг не распарсился, то считаем, что он не может быть применён
        if (!exp) {
          return acc;
        }

        const testid = expData.boxes[idx];
        const rawFlags = exp.flags;
        let flags;

        // У паспорта сложный способ передачи флагов (вместо готового JSON).
        // Парсим массив строк. Если в строке нет "=", значит значение true, иначе — строка после равно
        // Например, "web_supported_langs= [\"ru\", \"en\", \"tr\", \"kk\", \"uz\", \"az\", \"fr\", \"iw\", \"he\", \"pt\"]\n"
        if (Array.isArray(rawFlags)) {
          flags = rawFlags.reduce<Dict<unknown>>((acc, flagValue) => {
            const equalPos = flagValue.indexOf('=');
            // Если вдруг строка начинается с "=", то считаем что это имя (какой-то ненормальный крайний случай)
            const name = equalPos <= 0 ? flagValue : flagValue.substring(0, equalPos);
            let value: boolean | string | unknown = true;

            if (equalPos > 0) {
              const rawValue = flagValue.substring(equalPos + 1);

              // Пытаемся распарсить получившуюся строку-значение. Но если не получилось, то оставляем строку как есть
              value = safelyParseJSON(rawValue, rawValue);
            }

            acc[name] = value;

            return acc;
          }, {});
        } else {
          flags = rawFlags;
        }

        acc.flags = Object.assign(acc.flags, flags);
        acc.boxes.push(testid);

        return acc;
      },
      {
        flags: {},
        boxes: [],
      },
    );
  }

  /* eslint-disable max-len */
  // @see https://wiki.yandex-team.ru/jandekspoisk/kachestvopoiska/abt/uaas/
  // * Пример JSON-а эксперимента:
  // [
  //     {
  //       "HANDLER": "TRUST",
  //       "CONTEXT": {
  //         "TRUST": {
  //           "flag1": 1,
  //           "flag2": 2
  //         }
  //       },
  //       "CONDITION": ""
  //     }
  //   ]

  // [{
  //     "HANDLER": "TRUST",
  //     "CONTEXT": {
  //         "TRUST": {
  //             "preferred-method": true,
  //             "cvv-only": true,
  //             "card-system": true,
  //             "template": "checkout_sarah",
  //             "cvv-required": true
  //         }
  //     },
  //     "CONDITION": "env_test"
  // }]

  // curl -i "http://uaas.search.yandex.net" -H "Y-Service: trust" -H "X-Forwarded-For-Y: 0" -H "X-Yandex-RandomUID: 3343396901573058978" -H "X-Yandex-AppInfo: eyJ2ZXJzaW9uIjo2MTAsImRldmljZVR5cGUiOiJkZXNrdG9wIn0="
  // X-Yandex-ExpSplitParams: eyJyIjowLCJzIjoidHJ1c3QiLCJkIjoiZGVza3RvcCIsIm0iOiIiLCJiIjoiVW5rbm93biIsImkiOnRydWUsIm4iOiIiLCJmIjoia2lub3BvaXNrIn0=
  // X-Yandex-LogstatUID: 3343396901573058978
  // X-Yandex-ExpConfigVersion: 15675
  // X-Yandex-ExpBoxes: 273885,0,96
  // X-Yandex-ExpBoxes-Crypted: GzFkfkflEMrvu0Mt8iKvIg,,
  // X-Yandex-ExpFlags: W3siSEFORExFUiI6IlRSVVNUIiwiQ09OVEVYVCI6eyJUUlVTVCI6eyJFbmFibGVSdHhCeVN1cCI6dHJ1ZX19fV0=, W3siSEFORExFUiI6IlRSVVNUIiwiQ09OVEVYVCI6eyJUUlVTVCI6eyJFbmFibGVSdHhCeVN1cCI6dHJ1ZX19fV0=
  /* eslint-enable max-len */
  private async getFromUAAS(): Promise<ExperimentsRawData | undefined> {
    if (this.config.disabled) {
      return undefined;
    }

    const { yandexuid } = this.cookieService.getCookies();

    const { handlerName, abUrl } = this.config;
    const reqid = this.requestIdService.getRequestId();
    const uaTraits = await this.uatraitsService.getUatraits();
    const userHeaders = this.context.headers;

    try {
      const requestParams = {
        family: 6,
        headers: {
          Host: String(userHeaders.host),
          'Y-Service': handlerName.toLowerCase(),
          'X-Req-Id': reqid,
          'User-Agent': String(userHeaders['user-agent']),
          'X-Forwarded-For-Y': String(this.context.req.ip || '0'),
          // Добавить при выкатке в качестве виджетов на других сервисах
          // см https://clubs.at.yandex-team.ru/experiments/1850
          // Откуда брать правильное значение нужно придумать :)
          // 'X-Yandex-SourceService':
          'X-Yandex-RandomUID': yandexuid || '0',
          'X-Yandex-AppInfo': base64Encode(
            JSON.stringify({
              deviceType: uaTraits?.isTouch ? 'touch' : 'desktop',
            }),
          ),
        },
      };

      this.logger.debug('Request uaas for token with params', { reqid, params: requestParams });

      const { headers } = await this.http.get(abUrl, requestParams);

      const flagsData = this.extractEncodedFlagsInfo(headers);

      this.logger.log('Receive uaas for token', { reqid, flags: flagsData.flags });

      return flagsData;
    } catch (e) {
      this.logger.error("Can't get experiments", e);
    }

    return undefined;
  }

  private async getFromECOO(): Promise<ExperimentsRawData | undefined> {
    const { handlerName, abUrl, disabled, ecooUrl } = this.config;
    const { yexp: yexpCookie } = this.cookieService.getCookies();

    if (disabled || !yexpCookie) {
      return undefined;
    }

    try {
      const reqid = this.requestIdService.getRequestId();
      const requestEcooParams = {
        family: 6,
        params: {
          key: yexpCookie,
        },
      };

      this.logger.debug(`Request ecoo for user exps with "${yexpCookie}"`);

      /* Получаем залипания пользователя из куки
       * @see https://wiki.yandex-team.ru/serp/experiments/zerotesting/apidescription/#6
       **/
      const testids = (await this.http.post<string[]>(ecooUrl, {}, requestEcooParams)).data;

      const requestParams = {
        params: {
          'test-id': testids.join('_'),
        },
        family: 6,
      };

      this.logger.debug('Request uaas for testids', { testids });

      // Получаем данные этих тестидов
      const { headers } = await this.http.get(
        `${abUrl}/${handlerName.toLowerCase()}`,
        requestParams,
      );

      const flagsData = this.extractEncodedFlagsInfo(headers);

      this.logger.log('Receive uaas for testids', { reqid, testids, flags: flagsData.flags });

      return flagsData;
    } catch (e) {
      this.logger.error("Can't get experiments", e);
    }

    return undefined;
  }

  private extractEncodedFlagsInfo(headers: AxiosResponseHeaders) {
    return {
      flags: headers?.['x-yandex-expflags']?.split?.(',') || [],
      boxes: headers?.['x-yandex-expboxes']?.split?.(';') || [],
    };
  }
}
