import { createRequire } from 'module';
import path from 'path';

import geobase from '@yandex-int/yandex-geobase';

import { GeobaseLookupInterface, GeobaseOptions } from './interfaces';

const nativeRequire = createRequire(process.cwd());

const LOOKUP_METHODS: Array<keyof GeobaseLookupInterface> = [
  'calcDistance',
  'getAsNameByIp',
  'getAsTraitsByIp',
  'getChildrenIds',
  'getCountryId',
  'getCountryRegion',
  'getIpTraits',
  'getIspCodeByIp',
  'getKnownLinguistics',
  'getKnownServices',
  'getKnownTimezones',
  'getLinguistics',
  'getParentsIds',
  'getRegion',
  'getRegionById',
  'getRegionByIp',
  'getRegionIdByIp',
  'getRegionIdByLocation',
  'getRegionsByType',
  'getReliabilitiesByIp',
  'getTimezoneById',
  'getTimezoneByLocation',
  'getTimezoneByLongitude',
  'getTimezoneByName',
  'getTimezoneName',
  'haveService',
  'isIdInRegion',
  'isIpInRegion',
  'isTorIp',
  'makePinpointGeolocation',
];

type MockedData = Partial<Record<keyof GeobaseLookupInterface, unknown>>;

interface Lookup extends GeobaseLookupInterface {
  binding: GeobaseLookupInterface;
}

let instance: GeobaseLookupInterface | null = null;

export function createGeobaseLookup(options: GeobaseOptions): GeobaseLookupInterface {
  if (process.env.GEOBASE_MOCKDATA_PATH) {
    instance = createDevLookup(process.env.GEOBASE_MOCKDATA_PATH);
  }

  if (!instance) {
    const { dataPath } = options;

    instance = geobase.v6({ geobaseData: dataPath }) as unknown as GeobaseLookupInterface;
    // NOTE: В `@yandex-int/geo` некорректная реализация метода,
    // так как должно быть 2 аргумента, вместо одного
    instance.haveService = function (this: Lookup, id, serviceName) {
      return this.binding.haveService(id, serviceName);
    };
  }

  return instance;
}

function createDevLookup(mockDataPath: string): GeobaseLookupInterface {
  const data = loadMockData(mockDataPath);

  const lookup = new Proxy<GeobaseLookupInterface>({} as GeobaseLookupInterface, {
    get(target, key: keyof GeobaseLookupInterface) {
      if (LOOKUP_METHODS.includes(key)) {
        return () => {
          return data[key];
        };
      }

      return target[key];
    },
  });

  return lookup;
}

function loadMockData(mockDataPath: string): MockedData {
  const data = nativeRequire(path.resolve(process.cwd(), mockDataPath));

  return data;
}
