import { DynamicModule, Module, Provider } from '@nestjs/common';

import { createGeobaseLookup } from './geobase-lookup';
import {
  DEFAULT_GEOBASE_OPTIONS,
  GEOBASE_LOOKUP_TOKEN,
  GEOBASE_OPTIONS_TOKEN,
} from './geobase.constants';
import { GeobaseService } from './geobase.service';
import {
  GeobaseModuleAsyncOptions,
  GeobaseModuleOptions,
  GeobaseOptions,
  GeobaseOptionsFactory,
} from './interfaces';

@Module({
  providers: [
    GeobaseService,
    {
      provide: GEOBASE_OPTIONS_TOKEN,
      useValue: DEFAULT_GEOBASE_OPTIONS,
    },
    {
      provide: GEOBASE_LOOKUP_TOKEN,
      useFactory(options: GeobaseOptions) {
        return createGeobaseLookup(options);
      },
      inject: [GEOBASE_OPTIONS_TOKEN],
    },
  ],
  exports: [GeobaseService],
})
export class GeobaseModule {
  static forRoot(options: GeobaseModuleOptions) {
    const { isGlobal, ...geobaseOptions } = options;

    return {
      module: GeobaseModule,
      global: isGlobal,
      providers: [{ provide: GEOBASE_OPTIONS_TOKEN, useValue: geobaseOptions }],
    };
  }

  static forRootAsync(options: GeobaseModuleAsyncOptions): DynamicModule {
    return {
      module: GeobaseModule,
      global: options.isGlobal,
      imports: options.imports,
      providers: this.createAsyncProviders(options),
    };
  }

  private static createAsyncProviders(options: GeobaseModuleAsyncOptions): Provider[] {
    const providers: Provider[] = [this.createAsyncOptionsProvider(options)];

    if (!options.useExisting && options.useClass) {
      providers.push({ provide: options.useClass, useClass: options.useClass });
    }

    return providers;
  }

  private static createAsyncOptionsProvider(options: GeobaseModuleAsyncOptions): Provider {
    const { useFactory, useExisting, useClass, inject = [] } = options;

    if (useFactory) {
      return {
        provide: GEOBASE_OPTIONS_TOKEN,
        useFactory,
        inject,
      };
    }

    const dep = useExisting || useClass;

    if (dep) {
      return {
        provide: GEOBASE_OPTIONS_TOKEN,
        async useFactory(optionsFactory: GeobaseOptionsFactory) {
          return await optionsFactory.createGeobaseOptions();
        },
        inject: [dep],
      };
    }

    throw new Error('Invalid configuration. Must provide useFactory, useClass or useExisting');
  }
}
