import { Inject, Injectable } from '@nestjs/common';

import { GEOBASE_LOOKUP_TOKEN } from './geobase.constants';
import {
  GeobaseAsTraits,
  GeobaseGeolocation,
  GeobaseGeolocationSearchData,
  GeobaseIpTraits,
  GeobaseLinguisticsItem,
  GeobaseLookupInterface,
  GeobaseRegion,
  GeobaseRegionData,
  GeobaseRegionType,
  GeobaseReliabilityRegion,
  GeobaseTimezone,
  GeobaseTimezoneExtendedInfo,
} from './interfaces';

@Injectable()
export class GeobaseService {
  private lookup: GeobaseLookupInterface;

  constructor(@Inject(GEOBASE_LOOKUP_TOKEN) lookup: GeobaseLookupInterface) {
    this.lookup = lookup;
  }

  calcDistance(lat1: number, lon1: number, lat2: number, lon2: number) {
    return this.lookup.calcDistance(lat1, lon1, lat2, lon2);
  }

  getAsNameByIp(ip: string) {
    return this.lookup.getAsNameByIp(ip);
  }

  getAsTraitsByIp(ip: string): GeobaseAsTraits {
    return this.lookup.getAsTraitsByIp(ip);
  }

  getChildrenIds(id: number, domain?: string) {
    return this.lookup.getChildrenIds(id, domain) || [];
  }

  getCountryId(id: number, domain?: string) {
    return this.lookup.getCountryId(id, domain);
  }

  getCountryRegion(regionId: number): GeobaseRegion {
    return this.lookup.getCountryRegion(regionId);
  }

  getIpTraits(ip: string): GeobaseIpTraits {
    return this.lookup.getIpTraits(ip);
  }

  getIspCodeByIp(ip: string) {
    return this.lookup.getIspCodeByIp(ip);
  }

  getKnownLinguistics() {
    return this.lookup.getKnownLinguistics();
  }

  getKnownServices() {
    return this.lookup.getKnownServices();
  }

  getKnownTimezones(): GeobaseTimezoneExtendedInfo[] {
    return this.lookup.getKnownTimezones();
  }

  getLinguistics(id: number, language: string): GeobaseLinguisticsItem {
    return this.lookup.getLinguistics(id, language);
  }

  getParentsIds(id: number, domain?: string) {
    return this.lookup.getParentsIds(id, domain) || [];
  }

  getRegion(data: GeobaseRegionData) {
    const { yandexGid, xForwardedFor, xRealIp } = data;

    if (yandexGid || xForwardedFor || xRealIp) {
      return this.lookup.getRegion(yandexGid, xForwardedFor, xRealIp);
    }

    return undefined;
  }

  getRegionById(id: number, domain?: string): GeobaseRegion | undefined {
    return this.lookup.getRegionById(id, domain);
  }

  getRegionByIp(ip: string, domain?: string): GeobaseRegion | undefined {
    return this.lookup.getRegionByIp(ip, domain);
  }

  getRegionIdByIp(ip: string) {
    return this.lookup.getRegionIdByIp(ip);
  }

  getRegionIdByLocation(lat: number, lon: number, isTune?: boolean) {
    return this.lookup.getRegionIdByLocation(lat, lon, isTune);
  }

  getRegionsByType(type: GeobaseRegionType): GeobaseRegion[] {
    return this.lookup.getRegionsByType(type);
  }

  getReliabilitiesByIp(ip: string): GeobaseReliabilityRegion[] {
    return this.lookup.getReliabilitiesByIp(ip);
  }

  getTimezoneById(id: number): GeobaseTimezone {
    return this.lookup.getTimezoneById(id);
  }

  getTimezoneByLocation(lat: number, lon: number): GeobaseTimezone {
    return this.lookup.getTimezoneByLocation(lat, lon);
  }

  getTimezoneByLongitude(lon: number): GeobaseTimezone {
    return this.lookup.getTimezoneByLongitude(lon);
  }

  getTimezoneByName(name: string): GeobaseTimezone {
    return this.lookup.getTimezoneByName(name);
  }

  getTimezoneName(id: number) {
    return this.lookup.getTimezoneName(id);
  }

  haveService(id: number, serviceName: string) {
    return this.lookup.haveService(id, serviceName);
  }

  isIdInRegion(id: number, pid: number, domain?: string) {
    return this.lookup.isIdInRegion(id, pid, domain);
  }

  isIpInRegion(ip: string, id: number, domain?: string) {
    return this.lookup.isIpInRegion(ip, id, domain);
  }

  isTorIp(ip: string) {
    return this.lookup.isTorIp(ip);
  }

  makePinpointGeolocation(
    searchData: GeobaseGeolocationSearchData,
    ypCookie: string,
    ysCookie: string,
  ): GeobaseGeolocation {
    return this.lookup.makePinpointGeolocation(searchData, ypCookie, ysCookie);
  }
}
