import http from 'http';

import { AxiosInstance, AxiosRequestConfig, AxiosResponse } from 'axios';
import { errorLogger, requestLogger, responseLogger } from 'axios-logger';

import { Inject, Injectable } from '@nestjs/common';
import { BlackboxService } from '@server/shared/blackbox';
import { LoggerService } from '@server/shared/logger';
import { TvmService } from '@yandex-int/nest-infra';

import { AXIOS_INSTANCE_TOKEN, AXIOS_OPTIONS_TOKEN } from './http.constants';
import { HttpConfig } from './http.interface';

export type RequestConfig<D> = Omit<AxiosRequestConfig<D>, 'headers'> & {
  headers?: Record<string, string | number | boolean | undefined>;
};

/** Манкипатчим глобальный http агент, чтобы форсить всегда ipv6, даже в запросах, которыми мы не управляем  */
// @ts-expect-error Утверждается, что addRequest нет в прототипе. Но ТС врёт
const addRequest = http.Agent.prototype.addRequest;

// @ts-expect-error аналогично
http.Agent.prototype.addRequest = function (request, options) {
  options.family = 6;

  return addRequest.call(this, request, options);
};

@Injectable()
export class HttpService {
  constructor(
    @Inject(AXIOS_INSTANCE_TOKEN)
    private readonly instance: AxiosInstance,
    @Inject(AXIOS_OPTIONS_TOKEN)
    private readonly config: HttpConfig,
    private tvmService: TvmService,
    private blackboxService: BlackboxService,
    private logger: LoggerService,
  ) {
    this.registerInterceptors();
  }

  get axiosRef(): AxiosInstance {
    return this.instance;
  }

  get<T = any, U = AxiosResponse<T>, D = any>(url: string, config?: RequestConfig<D>): Promise<U> {
    return this.instance.get(url, config as AxiosRequestConfig<D>);
  }

  post<T = any, U = AxiosResponse<T>, D = any>(
    url: string,
    data?: D,
    config?: RequestConfig<D>,
  ): Promise<U> {
    return this.instance.post(url, data, config as AxiosRequestConfig<D>);
  }

  delete<T = any, U = AxiosResponse<T>, D = any>(
    url: string,
    config?: RequestConfig<D>,
  ): Promise<U> {
    return this.instance.delete(url, config as AxiosRequestConfig<D>);
  }

  put<T = any, U = AxiosResponse<T>, D = any>(
    url: string,
    data?: D,
    config?: RequestConfig<D>,
  ): Promise<U> {
    return this.instance.put(url, data, config as AxiosRequestConfig<D>);
  }

  private registerInterceptors() {
    this.axiosRef.interceptors.request.use(async (config) => {
      const { tvmTargetName, useUserTicket } = this.config;
      const tvm = tvmTargetName ? await this.tvmService.getTicket(tvmTargetName) : null;
      const blackbox = await this.blackboxService.getBlackbox();

      if (!config.headers) {
        config.headers = {};
      }

      if (tvm?.ticket) {
        config.headers['X-Ya-Service-Ticket'] = tvm.ticket;
      }

      if (useUserTicket && blackbox?.userTicket) {
        config.headers['X-Ya-User-Ticket'] = blackbox.userTicket;
      }

      return config;
    });

    this.axiosRef.interceptors.request.use((request) => {
      // TODO: Отключаем логирование данных, т.к. в данные может попасть чувствительная информация,
      // в будущем нужно добавить blacklist с нужными полями.
      return requestLogger(request, {
        data: false,
        params: true,
        dateFormat: 'dd.mm.yyyy HH:MM:ss',
        logger: this.logger.log,
      });
    }, errorLogger);
    this.axiosRef.interceptors.response.use((response) => {
      return responseLogger(response, {
        data: false,
        params: true,
        dateFormat: 'dd.mm.yyyy HH:MM:ss',
        logger: this.logger.log,
      });
    }, errorLogger);

    this.axiosRef.interceptors.response.use(
      (response) => response,
      (error) => {
        // https://github.com/axios/axios#response-schema
        error?.config?.headers && delete error.config.headers;
        error?.request?._header && delete error.request._header;

        return Promise.reject(error);
      },
    );
  }
}
