import { Injectable } from '@nestjs/common';
import { ApplicationConfigService, LoggerConfigService } from '@server/config';
import { Dict } from '@shared/types/common';
import { YEnv } from '@shared/types/env';
import { Context, RequestIdService } from '@yandex-int/nest-common';
import createLogger, { StreamOptions, Logger as YandexLogger } from '@yandex-int/yandex-logger';
import blackbox from '@yandex-int/yandex-logger/middleware/blackbox';
import err from '@yandex-int/yandex-logger/middleware/err';
import req from '@yandex-int/yandex-logger/middleware/req';
import requestId from '@yandex-int/yandex-logger/middleware/request-id';
import securePassportCookie from '@yandex-int/yandex-logger/middleware/secure-passport-cookie';
import deployStream from '@yandex-int/yandex-logger/streams/deploy';
import lineStream from '@yandex-int/yandex-logger/streams/line';

function isError(data: Error | unknown): data is Error {
  return Boolean((data as Error)?.stack);
}

interface ParsedReq {
  method?: 'GET' | 'POST' | 'PUT';
  url?: string;
  headers?: Dict<string>;
  body?: string | unknown;
  remoteAddress?: string;
  remotePort?: number;
  cookies?: Dict<string>;
  yandexuid?: string;
}

interface ILogRecord {
  pid?: number;
  date?: Date;
  level?: number;
  levelName?: 'WARN';
  req?: ParsedReq;
  /** Полный урл текущего (обрабатываемого в данный момент) запроса */
  url?: string;
  /** Объект ошибки следует передавать так */
  err?: Error;
  /** 'duffman-access-log' */
  source?: string;
  msgFormat?: string;

  msgArgs?: [Error | unknown];
  msg?: string;
  name?: 'yandex-logger';
  /** Версия приложения */
  version?: string;
  /** Окружение */
  env?: YEnv;
  reqid?: string;
  /** Данные пользователя для логгирования */
  user?: {
    uid?: string;
    login?: string;
    email?: string;
    lang?: string;
    realIp?: string;
  };
  /** IP пользователя === user.realIp */
  ip?: string;
  referrer?: string;
  /** Агент из заголовков */
  useragent?: string;
  /** Имя хоста обработчика запроса: os.hostname() (не из урла запроса) */
  hostname?: string;
  yandexuid?: string;
}

/**
 * Полезные ссылки:
 * https://github.yandex-team.ru/search-interfaces/frontend/blob/master/packages/yandex-logger/docs/stream-error-booster.md#пример-подключения-на-примере-проекта-tap-backend-в-deploy
 */
@Injectable()
export class LoggerService {
  private logger: YandexLogger;

  constructor(
    private appConfig: ApplicationConfigService,
    private loggerConfig: LoggerConfigService,
    private context?: Context,
    private requestIdService?: RequestIdService,
  ) {
    const logType = this.loggerConfig.options.logType;

    this.logger = createLogger({
      fields: {
        version: this.appConfig.version,
        env: this.appConfig.yenv,
      },
      middleware: [err(), requestId(), blackbox(), this.customReqHandler(), securePassportCookie()],
      streams: [
        logType === 'deploy' && {
          level: 'debug' as const,
          stream: deployStream(),
        },
        logType === 'strLine' && {
          level: this.loggerConfig.options.logLevel,
          stream: lineStream({
            stream: process.stdout,
            template: [
              // Оставлено специально на случай отладки логов
              // '{{debug}}',
              '{{levelName}}{{#reqid}}({{reqid}}){{/reqid}}: {{msg}} {{args}}',
              '{{#err.stack}}\n{{err.stack}}{{/err.stack}}',
            ].join(''),
            resolvers: {
              // debug: (record: ILogRecord) => {
              //   console.log('record -------->', record);

              //   return '';
              // },
              reqid: (record: ILogRecord) => {
                return record.reqid || '';
              },
              args: (record: ILogRecord) => {
                const args = record?.msgArgs?.filter((item) => item !== undefined);

                if (!args?.length) {
                  return '';
                }

                return args.map((arg) => JSON.stringify(arg)).join(' ');
              },
            },
          }),
        },
      ].filter(Boolean) as StreamOptions[],
    });

    this.debug = this.logger.debug.bind(this.logger);
    this.log = this.logger.info.bind(this.logger);
    this.info = this.logger.info.bind(this.logger);
    this.warn = this.logger.warn.bind(this.logger);
    this.error = this.logger.error.bind(this.logger);
  }

  log: YandexLogger['info'];
  info: YandexLogger['info'];
  warn: YandexLogger['warn'];
  debug: YandexLogger['debug'];
  error: YandexLogger['error'];

  /**
   * Оборачиваем готовый мидлвар req, досыпая нужные нам параметры
   * @returns {void}
   */
  customReqHandler() {
    return (record: ILogRecord): void => {
      if (!record.req && this.context) {
        // @ts-expect-error в запись должен попасть сырой запрос, чтобы распарситься. Входной тип не совпадает с выходным
        record.req = this.context.req;
      }
      req()(record);

      if (!record.reqid && this.requestIdService) {
        record.reqid = this.requestIdService.getRequestId();
      }

      if (record.user?.realIp) {
        record.ip = record.user?.realIp;
      }

      if (isError(record.msgArgs?.[0])) {
        record.err = record.msgArgs?.[0];
      }

      if (record.req && record.yandexuid && !record?.req?.yandexuid) {
        record.req.yandexuid = record.yandexuid;
      }
    };
  }
}

export const staticLogger = new LoggerService(
  new ApplicationConfigService(),
  new LoggerConfigService(),
);
