import { NextFunction, Request, Response } from 'express';

import { Injectable, NestMiddleware } from '@nestjs/common';
import { ApplicationConfigService } from '@server/config';
import { PathsConfigService } from '@server/config/services';
import { BlackboxService } from '@server/shared//blackbox';
import { ExperimentsService } from '@server/shared/experiments';
import { LangDetectService } from '@server/shared/lang-detect';
import { isProdLike } from '@shared/helpers/isProdLike';
import { IpregService } from '@yandex-id/ipreg';
import { TldService } from '@yandex-int/nest-common';

@Injectable()
export class EnterGatewayMiddleware implements NestMiddleware {
  // eslint-disable-next-line max-params
  constructor(
    private blackboxService: BlackboxService,
    private langDetectService: LangDetectService,
    private ipregService: IpregService,
    private applicationConfig: ApplicationConfigService,
    private experimentsService: ExperimentsService,
    private tldService: TldService,
    private pathsConfig: PathsConfigService,
  ) {}

  async use(req: Request, res: Response, next: NextFunction) {
    // Сплиты в письмах "почему то" присылают ссылку passport.yandex.ru/?showSplit=1 вместо passport.yandex.ru/profile?showSplit=1
    // Подробности я в https://lego-team.slack.com/archives/C02SPTGJNBS/p1657804412881869
    // Остальные параметры пробрасываем на случай, если не только сплит что-то подобное делает
    if (req.query.showSplit) {
      return res.redirect(this.getPassportProfileUrl(req));
    }

    // пускаем
    if (
      // либо пользователей под флагом,
      (await this.isUserInExperiment()) ||
      // либо в тестинге
      !isProdLike(this.applicationConfig.yenv) ||
      // либо авторизованных яндексоидов
      (await this.blackboxService.isYandexoid()) ||
      // либо пользователей из внутренней сети
      this.ipregService.isUserNet()
    ) {
      return next();
    }

    res.redirect(this.getPassportProfileUrl(req));
  }

  async isUserInExperiment() {
    const { flags } = await this.experimentsService.getExperiments();

    return (
      flags['passport-new-lk-exp'] &&
      flags['passport-new-navigation-exp'] &&
      this.tldService.getTld() === 'ru' &&
      this.langDetectService.userLang === 'ru'
    );
  }

  getPassportProfileUrl(req: Request) {
    const searchParams = req.originalUrl.split('?')[1];
    const redirectUrl = searchParams
      ? `${this.pathsConfig.passportProfileUrl}?${searchParams}`
      : this.pathsConfig.passportProfileUrl;

    return redirectUrl;
  }
}
