import { coerce, gte } from 'semver';

import { Injectable } from '@nestjs/common';
import { GraphQLSchemaHost } from '@nestjs/graphql';
import { ApplicationConfigService, YandexPayConfigService } from '@server/config';
import { ViewerService } from '@server/features/viewer';
import { CSPService } from '@server/shared/csp';
import { ExperimentsService } from '@server/shared/experiments';
import { TrustService } from '@server/shared/trust';
import { NextBridge, NextPageContext, ServerSideProps } from '@shared/types/next-bridge';
import { RequestIdService, TldService } from '@yandex-int/nest-common';
import { SecretkeyService, UatraitsService } from '@yandex-int/nest-infra';

@Injectable()
export class NextBridgeService implements NextBridge {
  // eslint-disable-next-line max-params
  constructor(
    private tldService: TldService,
    private appConfig: ApplicationConfigService,
    private graphQLSchemaHost: GraphQLSchemaHost,
    private viewerService: ViewerService,
    private secretkeyService: SecretkeyService,
    private cspService: CSPService,
    private uatraitsService: UatraitsService,
    private trustService: TrustService,
    private requestIdService: RequestIdService,
    private experimentsService: ExperimentsService,
    private yandexPayConfig: YandexPayConfigService,
  ) {}

  async getServerSideProps(context?: NextPageContext): Promise<ServerSideProps> {
    const tld = this.tldService.getTld() ?? 'ru';
    const viewer = await this.viewerService.getCommonUserData();
    const plusBalance = await this.trustService.getPlusBalance();
    const secretKey = await this.secretkeyService.getSecretkey();
    const ua = await this.uatraitsService.getUatraits();
    const exp = await this.experimentsService.getExperiments();

    const result: ServerSideProps = {
      reqid: this.requestIdService.getRequestId(),
      version: this.appConfig.version,
      secretKey: secretKey ?? '',
      tld: tld ?? 'ru',
      exp,
      yenv: this.appConfig.yenv,
      viewer: {
        uid: viewer.uid,
        avatarId: viewer.avatarId,
        havePlus: viewer.havePlus,
        plusBalance: plusBalance,
        isFamilyAdmin: viewer.isFamilyAdmin,
        publicId: viewer.hasPublicIdSet ? viewer.publicId : '',
      },
      language: {
        id: viewer.language,
        // TODO: Использовать нормальный мапинг языка.
        name: capitalizeFirstLetter(viewer.language),
      },
      ua: {
        isTouch: ua?.isTouch,
        isMobile: ua?.isMobile && !ua?.isTablet,
        mobileOsFamily:
          // eslint-disable-next-line no-nested-ternary
          ua?.OSFamily === 'Android' ? 'android' : ua?.OSFamily === 'iOS' ? 'ios' : undefined,
        isYandexApp: ua?.BrowserName === 'YandexSearch',
        browserVersion: ua?.BrowserVersion,
        osVersion: ua?.OSVersion,
        isSearchApp: Boolean(context?.query?.isSearchApp),
      },
      uiFlags: {},
      yandexPayConfig: this.yandexPayConfig.options,
    };

    if (ua?.OSFamily === 'iOS' && ua.OSVersion) {
      const transformedOsVersion = coerce(ua.OSVersion);
      const transformedConstVersion = coerce('13');

      if (transformedOsVersion && transformedConstVersion) {
        result.uiFlags.additionalNavigation = gte(transformedOsVersion, transformedConstVersion);
      }
    } else {
      result.uiFlags.additionalNavigation = true;
    }

    if (viewer.isYandexNet) {
      result.isYandexNet = true;
    }

    if (viewer.isYandexoid) {
      result.isYandexoid = true;
    }

    return result;
  }

  get graphqlSchema() {
    return this.graphQLSchemaHost.schema;
  }

  getNonce() {
    const nonce = this.cspService.getNonce();

    return nonce;
  }
}

function capitalizeFirstLetter(value: string) {
  return value.charAt(0).toUpperCase() + value.slice(1);
}
