import { Server } from 'http';

import { RenderModule, RendererConfig } from 'nest-next';
import Next from 'next';
import type { NextServerOptions } from 'next/dist/server/next';

import { DynamicModule, Global, Module, OnModuleDestroy, OnModuleInit } from '@nestjs/common';
import { HttpAdapterHost } from '@nestjs/core';
import { ExpressAdapter } from '@nestjs/platform-express';
import { HmrCacheable, getStorageFor } from '@server/shared/hmr-storage';

import { patchNextLogger } from './patch-next-logger';

patchNextLogger();

export interface NextModuleOptions extends Partial<RendererConfig> {
  serverOptions?: NextServerOptions;
}

const UPGRADE_LISTENERS_KEY = 'UPGRADE_LISTENERS';
const RENDER_MODULE_KEY = 'RENDER_MODULE';

@Global()
@Module({
  exports: [RenderModule],
})
export class NextModule implements OnModuleInit, OnModuleDestroy {
  static async forRootAsync(options: NextModuleOptions): Promise<DynamicModule> {
    const NextRenderModule = await this.getRenderModule(options);

    return {
      module: NextModule,
      imports: [NextRenderModule],
    };
  }

  @HmrCacheable(module, RENDER_MODULE_KEY)
  private static getRenderModule(options: NextModuleOptions) {
    const {
      serverOptions = {},
      viewsDir = null,
      passthrough404 = true,
      ...rendererOptions
    } = options;

    return RenderModule.forRootAsync(Next(serverOptions), {
      ...rendererOptions,
      viewsDir,
      passthrough404,
    });
  }

  constructor(private httpAdapterHost: HttpAdapterHost<ExpressAdapter>) {}

  onModuleInit() {
    if (module.hot) {
      const server = this.getHttpServer();
      const currentListeners = server.rawListeners('upgrade');
      const cachedListeners = this.getCachedUpgradeListeners();

      // при инициализации модуля добавляем обработчики для веб-сокетов
      // из прошлого запуска сервера, так как после hmr сервер перезапускается,
      // а инстанс next повторно не будет вешать эти обработчики.
      cachedListeners.forEach((listener) => {
        if (currentListeners.indexOf(listener) === -1) {
          server.on('upgrade', listener);
        }
      });
    }
  }

  onModuleDestroy() {
    if (module.hot) {
      const server = this.getHttpServer();
      const storage = getStorageFor(module);

      // Сохраняем обработчики в module.hot.data,
      // чтобы можно было их восстановить при рестарте сервера
      storage.set(UPGRADE_LISTENERS_KEY, server.rawListeners('upgrade'));
    }
  }

  private getHttpServer(): Server {
    return this.httpAdapterHost.httpAdapter.getHttpServer();
  }

  @HmrCacheable(module, UPGRADE_LISTENERS_KEY)
  private getCachedUpgradeListeners() {
    const server = this.getHttpServer();

    return server.rawListeners('upgrade') as Array<(...args: unknown[]) => void>;
  }
}
